/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security;

import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.annotations.Beta;
import com.sap.cloud.sdk.cloudplatform.thread.Property;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContext;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextListener;
import com.sap.cloud.security.token.SecurityContext;
import com.sap.cloud.security.token.Token;

import io.vavr.control.Option;
import io.vavr.control.Try;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;

/**
 * {@link ThreadContextListener} implementation reading the Basic Authentication Header from the incoming request and
 * storing it in the current {@link ThreadContext}.
 */
@AllArgsConstructor( access = AccessLevel.PACKAGE )
@NoArgsConstructor
@Slf4j
@Beta
public class SecurityContextThreadContextListener implements ThreadContextListener
{
    /**
     * The key of the stored BasicCredentials in the current {@link ThreadContext}.
     */
    public static final String PROPERTY_SECURITY_CONTEXT_TOKEN =
        SecurityContextThreadContextListener.class.getName() + ":securityContextToken";

    @Getter
    private final int priority = DefaultPriorities.SCP_CF_SECURITY_CONTEXT_DECORATOR;

    @Nullable
    private Token token;

    @Override
    public
        void
        afterInitialize( @Nonnull final ThreadContext threadContext, @Nullable final ThreadContext parentThreadContext )
    {
        final Try<Token> tokenFromPreset = Option.of(this.token).toTry();
        final Try<Token> tokenFromParent = tokenFromPreset.orElse(() -> getTokenFromThreadContext(parentThreadContext));
        final Try<Token> token = tokenFromParent.orElse(this::getTokenFromCurrentThread);

        threadContext.setPropertyIfAbsent(PROPERTY_SECURITY_CONTEXT_TOKEN, Property.ofTry(token));
    }

    @Nonnull
    private Try<Token> getTokenFromThreadContext( @Nullable final ThreadContext threadContext )
    {
        return Option
            .of(threadContext)
            .toTry()
            .flatMap(context -> context.<Token> getProperty(PROPERTY_SECURITY_CONTEXT_TOKEN))
            .map(Property::getValue);
    }

    @Nonnull
    private Try<Token> getTokenFromCurrentThread()
    {
        return Try
            .of(SecurityContext::getToken)
            .map(Objects::requireNonNull)
            .onFailure(e -> log.debug("Failed to get token from SecurityContext of current thread."));
    }

    @Override
    public
        void
        beforeDestroy( @Nonnull final ThreadContext threadContext, @Nullable final ThreadContext parentThreadContext )
    {
        threadContext.removeProperty(PROPERTY_SECURITY_CONTEXT_TOKEN);
    }
}
