/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security;

import java.security.interfaces.RSAPublicKey;
import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.auth0.jwt.JWT;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceConfiguration;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceConfiguration.CircuitBreakerConfiguration;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceConfiguration.TimeLimiterConfiguration;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceDecorator;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceIsolationMode;
import com.sap.cloud.sdk.cloudplatform.security.exception.AuthTokenAccessException;
import com.sap.cloud.sdk.cloudplatform.security.exception.TokenRequestFailedException;
import com.sap.cloud.security.xsuaa.tokenflows.XsuaaTokenFlows;

import io.vavr.control.Try;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Deprecated
class AuthTokenDecoderLegacy implements AuthTokenDecoder
{
    @Getter
    @Nonnull
    private final OAuth2TokenServiceCache tokenServiceCache;

    AuthTokenDecoderLegacy()
    {
        this(ScpCfAuthTokenFacade.DEFAULT_TOKEN_SERVICE_CACHE);
    }

    /**
     * Default constructor for the AuthTokenDecoder.
     *
     * @param serviceCache
     *            Optional parameter for {@code OAuth2ServiceProvider} reference. If {@code null}, then
     *            {@link OAuth2TokenServiceCache} will be instantiated.
     */
    public AuthTokenDecoderLegacy( @Nullable final OAuth2TokenServiceCache serviceCache )
    {
        tokenServiceCache = serviceCache != null ? serviceCache : OAuth2TokenServiceCache.create();
    }

    @Nonnull
    @Override
    public AuthToken decode( @Nonnull final String encodedJwt )
        throws AuthTokenAccessException
    {
        return decode(encodedJwt, null);
    }

    @Nonnull
    @Override
    public AuthToken decode( @Nonnull final String encodedJwt, @Nullable final String refreshToken )
        throws AuthTokenAccessException
    {
        final DecodedJWT decodedJWT =
            Try
                .of(() -> JWT.decode(encodedJwt))
                .getOrElseThrow(e -> new AuthTokenAccessException("Failed to decode access token.", e));
        final List<RSAPublicKey> verificationKeys = AuthTokenValidator.getVerificationPublicKeysForJwt(decodedJWT);

        final String targetAlgorithm = decodedJWT.getAlgorithm();
        final AuthTokenValidator authTokenValidator = new AuthTokenValidator(targetAlgorithm, verificationKeys);

        Optional<DecodedJWT> verifiedToken = authTokenValidator.verifyToken(encodedJwt);

        // fallback: use refresh token to request new access token and try to verify once more
        if( !verifiedToken.isPresent() && refreshToken != null ) {
            final String refreshedEncodedJwt = getRefreshedToken(encodedJwt, refreshToken);
            verifiedToken = authTokenValidator.verifyToken(refreshedEncodedJwt);
        }
        return verifiedToken
            .map(AuthToken::new)
            .orElseThrow(() -> new AuthTokenAccessException("Failed to valiadte access token."));
    }

    @Nonnull
    String getRefreshedToken( @Nonnull final String encodedJwt, @Nonnull final String refreshToken )
    {
        final ResilienceConfiguration resilienceConfiguration =
            ResilienceConfiguration
                .of(AuthTokenDecoderLegacy.class)
                .isolationMode(ResilienceIsolationMode.NO_ISOLATION)
                .timeLimiterConfiguration(TimeLimiterConfiguration.of().timeoutDuration(Duration.ofSeconds(6)))
                .circuitBreakerConfiguration(CircuitBreakerConfiguration.of().waitDuration(Duration.ofSeconds(6)));

        final Supplier<String> tokenSupplier =
            ResilienceDecorator
                .decorateSupplier(
                    () -> sendRefreshTokenRequestAndParseResponse(encodedJwt, refreshToken),
                    resilienceConfiguration);

        return tokenSupplier.get();
    }

    private
        String
        sendRefreshTokenRequestAndParseResponse( @Nonnull final String encodedJwt, @Nonnull final String refreshToken )
    {
        final DecodedJWT accessToken = JWT.decode(encodedJwt);
        final XsuaaTokenFlows tokenFlows =
            OAuth2ServiceProvider
                .builder()
                .tokenServiceCache(tokenServiceCache)
                .staticAccessToken(accessToken)
                .build()
                .getXsuaaTokenFlows();

        return Try
            .of(() -> tokenFlows.refreshTokenFlow().refreshToken(refreshToken).execute().getAccessToken())
            .onFailure(e -> log.debug("Failed for access token {} and refresh token {}.", encodedJwt, refreshToken, e))
            .getOrElseThrow(e -> new TokenRequestFailedException("Refresh JWT request failed", e));
    }
}
