/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.auth0.jwt.JWT;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.google.common.collect.Sets;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceConfiguration;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceDecorator;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceIsolationMode;
import com.sap.cloud.sdk.cloudplatform.security.exception.AuthTokenAccessException;
import com.sap.cloud.sdk.cloudplatform.security.exception.TokenRequestDeniedException;
import com.sap.cloud.sdk.cloudplatform.security.exception.TokenRequestFailedException;
import com.sap.cloud.security.config.OAuth2ServiceConfiguration;
import com.sap.cloud.security.config.Service;
import com.sap.cloud.security.config.cf.CFConstants;
import com.sap.cloud.security.config.cf.CFEnvironment;
import com.sap.cloud.security.token.Token;
import com.sap.cloud.security.token.validation.CombiningValidator;
import com.sap.cloud.security.token.validation.ValidationResult;
import com.sap.cloud.security.token.validation.Validator;
import com.sap.cloud.security.token.validation.validators.JwtValidatorBuilder;
import com.sap.cloud.security.xsuaa.tokenflows.XsuaaTokenFlows;

import io.vavr.control.Option;
import io.vavr.control.Try;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@RequiredArgsConstructor( access = AccessLevel.PACKAGE )
@Deprecated
class AuthTokenDecoderXsuaa implements AuthTokenDecoder
{
    private static final String MESSAGE_INVALID_REFRESH_TOKEN =
        "Failed to get access token: no valid refresh token found in response of user token flow. Please make sure to correctly bind your application to an OAuth2 compatible service instance.";

    @Getter
    @Nonnull
    private final List<CombiningValidator<Token>> tokenValidators;

    @Nonnull
    private static final List<CombiningValidator<Token>> defaultValidators = loadOauth2Validators();

    AuthTokenDecoderXsuaa()
    {
        this(defaultValidators);
    }

    AuthTokenDecoderXsuaa( @Nullable final OAuth2ServiceConfiguration oauth2Configuration )
    {
        this(
            Option
                .of(oauth2Configuration)
                .onEmpty(() -> log.warn("AuthTokenFacade instantiated without an OAuth2 configuration."))
                .map(AuthTokenDecoderXsuaa::getJwtValidator)
                .map(Collections::singletonList)
                .getOrElse(Collections::emptyList));
    }

    @Nonnull
    @Override
    public AuthToken decode( @Nonnull final String encodedJwt )
        throws AuthTokenAccessException
    {
        if( tokenValidators.isEmpty() ) {
            throw new AuthTokenAccessException("AuthTokenDecoder was instantiated without a token validator.");
        }

        // add suppressed failures but only throw it if all validation attempts fail
        final List<Throwable> suppressedExceptions = new ArrayList<>();

        // first attempt validation the Cloud Security library
        final Optional<AuthToken> validatedToken =
            tokenValidators
                .stream()
                .map(tokenValidator -> Try.of(() -> validateJwtWithSecurityLibrary(encodedJwt, tokenValidator)))
                .peek(tokenValidation -> tokenValidation.onFailure(suppressedExceptions::add))
                .peek(tokenValidation -> tokenValidation.onFailure(e -> log.debug("JWT validation attempt failed.", e)))
                .filter(Try::isSuccess)
                .findFirst()
                .map(Try::get)
                .map(AuthToken::new);

        return validatedToken.orElseThrow(() -> {
            final AuthTokenAccessException e = new AuthTokenAccessException("Failed to verify JWT bearer.");
            suppressedExceptions.forEach(e::addSuppressed);
            return e;
        });
    }

    private DecodedJWT validateJwtWithSecurityLibrary(
        @Nonnull final String encodedJwt,
        @Nonnull final Validator<Token> tokenValidator )
    {
        final Token token = Token.create(encodedJwt);
        final ValidationResult result = tokenValidator.validate(token);
        if( result.isValid() ) {
            return JWT.decode(encodedJwt);
        }
        throw new AuthTokenAccessException("The token is invalid: " + result.getErrorDescription());
    }

    @Nonnull
    private static List<CombiningValidator<Token>> loadOauth2Validators()
    {
        final List<OAuth2ServiceConfiguration> oauth2Configurations = loadOauth2ServiceConfigurations();
        if( oauth2Configurations.isEmpty() ) {
            log.debug("No OAuth2 validators were registered since no configuration could be loaded.");
        }

        return oauth2Configurations
            .stream()
            .map(conf -> Try.of(() -> getJwtValidator(conf)))
            .filter(validator -> validator.onFailure(e -> log.warn("Failed to load validator.", e)).isSuccess())
            .map(Try::get)
            .collect(Collectors.toList());
    }

    @Nonnull
    private static List<OAuth2ServiceConfiguration> loadOauth2ServiceConfigurations()
    {
        // resolve environment properties for CloudFoundry, handle potential parsing errors
        final Try<CFEnvironment> environment = Try.of(() -> {
            final com.sap.cloud.sdk.cloudplatform.ScpCfCloudPlatform platform =
                com.sap.cloud.sdk.cloudplatform.ScpCfCloudPlatform.getInstanceOrThrow();
            return CFEnvironment
                .getInstance(k -> K8sWorkarounds.getEnvironmentVariable(platform, k), System::getProperty);
        });

        if( environment.isSuccess() ) {

            // read all configurations from environment, consider all potential service plans
            final List<OAuth2ServiceConfiguration> configurations =
                Sets
                    .cartesianProduct(EnumSet.allOf(Service.class), EnumSet.allOf(CFConstants.Plan.class))
                    .stream()
                    .map(p -> environment.get().loadForServicePlan((Service) p.get(0), (CFConstants.Plan) p.get(1)))
                    .filter(Objects::nonNull)
                    .peek(
                        c -> log
                            .debug(
                                "Found {} service binding with plan {}, client id {} and application id {}.",
                                c.getService().getCFName(),
                                c.getProperty(CFConstants.SERVICE_PLAN),
                                c.getClientId(),
                                c.getProperty(CFConstants.XSUAA.APP_ID)))
                    .collect(Collectors.toList());

            if( configurations.isEmpty() ) {
                log.warn("Could not find any OAuth2 based service bindings.");
            }
            return configurations;
        }
        log.error("Failed to read environment data for OAuth2 based configurations.", environment.getCause());
        return Collections.emptyList();
    }

    @Nonnull
    static CombiningValidator<Token> getJwtValidator( @Nonnull final OAuth2ServiceConfiguration configuration )
    {
        final CombiningValidator<Token> result = JwtValidatorBuilder.getInstance(configuration).build();

        // see backlog item to remove this workaround: CLOUDECOSYSTEM-9585
        final String workaroundClass = "com.sap.cloud.security.token.validation.validators.JwtIssuerValidator";
        if( result.getValidators().removeIf(validator -> workaroundClass.equals(validator.getClass().getName())) ) {
            result.getValidators().add(new CustomJwtIssuerValidator(configuration));
        }

        return result;
    }

    Future<String> getRefreshToken( @Nonnull final DecodedJWT jwt, @Nullable final OAuth2TokenServiceCache cache )
    {

        final ResilienceConfiguration resilienceConfiguration =
            ResilienceConfiguration
                .of(ScpCfAuthTokenFacade.class)
                .isolationMode(ResilienceIsolationMode.NO_ISOLATION)
                .timeLimiterConfiguration(
                    ResilienceConfiguration.TimeLimiterConfiguration.of().timeoutDuration(Duration.ofSeconds(6)))
                .circuitBreakerConfiguration(
                    ResilienceConfiguration.CircuitBreakerConfiguration.of().waitDuration(Duration.ofSeconds(6)));

        final Callable<String> tokenCallable =
            ResilienceDecorator
                .decorateCallable(() -> sendTokenRequestAndParseResponse(jwt, cache), resilienceConfiguration);

        final FutureTask<String> tokenFuture = new FutureTask<>(tokenCallable);
        tokenFuture.run();
        return tokenFuture;
    }

    @Nonnull
    private String sendTokenRequestAndParseResponse(
        @Nonnull final DecodedJWT jwt,
        @Nullable final OAuth2TokenServiceCache tokenServiceCache )
        throws TokenRequestDeniedException,
            TokenRequestFailedException
    {
        final String authorizationBearer = jwt.getToken();
        final XsuaaTokenFlows tokenFlows =
            OAuth2ServiceProvider
                .builder()
                .tokenServiceCache(tokenServiceCache)
                .staticAccessToken(jwt)
                .build()
                .getXsuaaTokenFlows();

        return Try
            .of(() -> tokenFlows.userTokenFlow().token(authorizationBearer).execute().getRefreshToken())
            .onFailure(e -> log.debug("User token request failed for auth token {}.", authorizationBearer, e))
            .filter(Objects::nonNull, () -> new TokenRequestFailedException(MESSAGE_INVALID_REFRESH_TOKEN))
            .getOrElseThrow(e -> new TokenRequestFailedException("Refresh JWT request failed", e));
    }
}
