/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security;

import java.io.IOException;
import java.net.URI;
import java.security.KeyFactory;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.X509EncodedKeySpec;
import java.util.Base64;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.entity.ContentType;

import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTVerifier;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.collect.Lists;
import com.google.common.collect.Streams;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.json.JsonSanitizer;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpClientAccessor;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpEntityUtil;
import com.sap.cloud.sdk.cloudplatform.exception.ShouldNotHappenException;
import com.sap.cloud.sdk.cloudplatform.security.exception.AuthTokenAccessException;
import com.sap.cloud.sdk.cloudplatform.security.exception.TokenRequestFailedException;

import io.vavr.control.Option;
import io.vavr.control.Try;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Deprecated
class AuthTokenValidator
{
    // only visible for testing
    static Caffeine<Object, Object> tokenCacheBuilder =
        Caffeine.newBuilder().maximumSize(100000).expireAfterAccess(5, TimeUnit.MINUTES);

    // only visible for testing
    static Cache<URI, List<String>> publicTokenKeyCache = tokenCacheBuilder.build();

    private final List<JWTVerifier> verifiers;

    /**
     * Create a new instance of {@link AuthTokenValidator}, which manages instances of {@link JWTVerifier}.
     *
     * @param tokenAlgorithm
     *            The token algorithm identifier.
     * @param verificationKeys
     *            List of {@link RSAPublicKey} to create verifiers from.
     */
    AuthTokenValidator( @Nonnull final String tokenAlgorithm, @Nonnull final List<RSAPublicKey> verificationKeys )
    {
        verifiers =
            verificationKeys
                .stream()
                .map(publicKey -> tryGetVerifierFromAlgorithm(publicKey, tokenAlgorithm))
                .filter(Try::isSuccess)
                .map(Try::get)
                .collect(Collectors.toList());
    }

    /**
     * Validate a provided token.
     *
     * @param encodedJwt
     *            The token to be verified.
     * @return Optionally a verified token. Might be altered due to the validation process. If the process failed for
     *         all verifiers, then this {@link Optional} remains empty.
     */
    @Nonnull
    Optional<DecodedJWT> verifyToken( @Nonnull final String encodedJwt )
    {
        return verifiers
            .stream()
            .map(verifier -> Try.of(() -> verifier.verify(encodedJwt)))
            .filter(tryToken -> tryToken.onFailure(e -> log.debug("Failed token verification.", e)).isSuccess())
            .findFirst()
            .map(Try::get);
    }

    /**
     * Static helper method to resolve all public keys which can be used to validate the provided JWT.
     *
     * @param decodedJwt
     *            The token for which public keys will be resolved.
     * @return A list of potentially matching {@link RSAPublicKey}.
     * @throws AuthTokenAccessException
     *             When the public keys could not be queried.
     */
    @Nonnull
    static List<RSAPublicKey> getVerificationPublicKeysForJwt( @Nonnull final DecodedJWT decodedJwt )
        throws AuthTokenAccessException
    {
        final JsonObject xsuaaServiceCredentials =
            Try
                .of(() -> getCloudPlatform().getXsuaaServiceCredentials(decodedJwt))
                .getOrElseThrow(e -> new AuthTokenAccessException("Failed to verify JWT bearer.", e));

        final List<String> publicKeysRaw = getPublicKeysFromCredentials(xsuaaServiceCredentials);

        return publicKeysRaw
            .stream()

            // refine String representation of RSA format
            .map(keyRaw -> keyRaw.replaceAll("\\n", ""))
            .map(keyRaw -> keyRaw.replace("-----BEGIN PUBLIC KEY-----", ""))
            .map(keyRaw -> keyRaw.replace("-----END PUBLIC KEY-----", ""))

            // derive security specification
            .map(keyRefined -> Try.of(() -> new X509EncodedKeySpec(Base64.getDecoder().decode(keyRefined))))

            // create a public key
            .map(trySpec -> trySpec.mapTry(keySpec -> KeyFactory.getInstance("RSA").generatePublic(keySpec)))
            .map(tryKey -> tryKey.mapTry(RSAPublicKey.class::cast))

            // filter for correct public keys
            .filter(tryKey -> tryKey.onFailure(e -> log.warn("Failed to parse public key.", e)).isSuccess())
            .map(Try::get)
            .collect(Collectors.toList());
    }

    @Nonnull
    private static Try<JWTVerifier> tryGetVerifierFromAlgorithm(
        @Nonnull final RSAPublicKey verificationKey,
        @Nullable final String algorithmIdentifier )
    {
        if( algorithmIdentifier == null ) {
            return Try
                .failure(
                    new AuthTokenAccessException(
                        "Failed to verify JWT bearer: no algorithm specified in token header."));
        }

        // limit the supported algorithms to RSA with different hash lengths to avoid the risks described at:
        // https://auth0.com/blog/critical-vulnerabilities-in-json-web-token-libraries/

        final Algorithm algorithm;
        switch( algorithmIdentifier ) {
            case "RS256":
                algorithm = Algorithm.RSA256(verificationKey, null);
                break;
            case "RS384":
                algorithm = Algorithm.RSA384(verificationKey, null);
                break;
            case "RS512":
                algorithm = Algorithm.RSA512(verificationKey, null);
                break;
            default:
                return Try
                    .failure(
                        new AuthTokenAccessException(
                            "Failed to verify JWT bearer: algorithm '" + algorithmIdentifier + "' not supported."));
        }

        return Try
            .of(() -> JWT.require(algorithm).build())
            .onFailure(e -> log.debug("Failed to instantiate token validator from algorithm.", e));
    }

    private static com.sap.cloud.sdk.cloudplatform.ScpCfCloudPlatform getCloudPlatform()
    {
        final com.sap.cloud.sdk.cloudplatform.CloudPlatform cloudPlatform =
            com.sap.cloud.sdk.cloudplatform.CloudPlatformAccessor.getCloudPlatform();

        if( !(cloudPlatform instanceof com.sap.cloud.sdk.cloudplatform.ScpCfCloudPlatform) ) {
            throw new ShouldNotHappenException(
                "The current Cloud platform is not an instance of "
                    + com.sap.cloud.sdk.cloudplatform.ScpCfCloudPlatform.class.getSimpleName()
                    + ". Please make sure to specify a dependency to com.sap.cloud.sdk.cloudplatform:core-scp-cf.");
        }

        return (com.sap.cloud.sdk.cloudplatform.ScpCfCloudPlatform) cloudPlatform;
    }

    private static List<String> getPublicKeysFromCredentials( final JsonObject xsuaaServiceCredentials )
        throws AuthTokenAccessException
    {
        final List<String> combinedResult = Lists.newArrayList();

        // try to load remote public keys
        final Try<List<String>> remoteKeys = Try.of(() -> getCachedRemotePublicKeys(xsuaaServiceCredentials));
        remoteKeys.onFailure(e -> log.warn("Failed to load remote public keys.", e)).onSuccess(combinedResult::addAll);

        // try to load local public key
        final Try<String> localKey = Try.of(() -> getLocalPublicKey(xsuaaServiceCredentials));
        localKey.onFailure(e -> log.warn("Failed to load local public key.", e)).onSuccess(combinedResult::add);

        if( combinedResult.isEmpty() ) {
            throw new AuthTokenAccessException(
                "Unable to resolve any public keys from local environment or remote endpoints.");
        }

        return combinedResult;
    }

    private static String getLocalPublicKey( final JsonObject xsuaaServiceCredentials )
        throws AuthTokenAccessException
    {
        return Option
            .of(xsuaaServiceCredentials.get("verificationkey"))
            .map(JsonElement::getAsString)
            .getOrElseThrow(
                () -> new AuthTokenAccessException(
                    "Failed to verify JWT bearer: no verification key found in XSUAA service credentials."));
    }

    private static List<String> getCachedRemotePublicKeys( final JsonObject xsuaaServiceCredentials )
        throws IOException
    {
        final String xsuaaUrl = xsuaaServiceCredentials.get("url").getAsString();
        final URI uri = URI.create((xsuaaUrl.endsWith("/") ? xsuaaUrl : xsuaaUrl + "/") + "token_keys");

        @Nullable
        List<String> cachedKeys = publicTokenKeyCache.getIfPresent(uri);
        if( cachedKeys == null ) {
            cachedKeys = getRemotePublicKeysFromUri(uri);
            publicTokenKeyCache.put(uri, cachedKeys);
        }
        return cachedKeys;
    }

    private static List<String> getRemotePublicKeysFromUri( final URI uri )
        throws IOException
    {
        final HttpGet tokenRequest = new HttpGet(uri);
        tokenRequest.setHeader(org.apache.http.HttpHeaders.ACCEPT, ContentType.APPLICATION_JSON.toString());

        final HttpClient httpClient = HttpClientAccessor.getHttpClientFactory().createHttpClient();
        final HttpResponse response = httpClient.execute(tokenRequest);
        final int statusCode = response.getStatusLine().getStatusCode();
        if( statusCode >= 400 && statusCode <= 599 ) {
            throw new TokenRequestFailedException(
                "Refresh JWT request failed with status code "
                    + statusCode
                    + ": "
                    + response.getStatusLine().getReasonPhrase());
        }

        final String responseBody = HttpEntityUtil.getResponseBody(response);
        final JsonObject responseJson = JsonParser.parseString(JsonSanitizer.sanitize(responseBody)).getAsJsonObject();
        final Iterable<JsonElement> keys = responseJson.getAsJsonArray("keys");

        return Streams
            .stream(keys)
            .map(AuthTokenValidator::tryParseRemotePublicKey)
            .filter(Try::isSuccess)
            .map(Try::get)
            .collect(Collectors.toList());
    }

    private static Try<String> tryParseRemotePublicKey( @Nullable final JsonElement keyEntry )
    {
        if( keyEntry == null ) {
            return Try.failure(new IOException("Detected missing key value."));
        }

        return Try
            .of(() -> keyEntry.getAsJsonObject().getAsJsonPrimitive("value").getAsString())
            .onFailure(e -> log.debug("Failed to read value for public key.", e));
    }
}
