/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security;

import java.net.URI;
import java.util.List;
import java.util.function.Predicate;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.sap.cloud.security.config.OAuth2ServiceConfiguration;
import com.sap.cloud.security.token.Token;
import com.sap.cloud.security.token.TokenClaims;
import com.sap.cloud.security.token.validation.ValidationResult;
import com.sap.cloud.security.token.validation.ValidationResults;
import com.sap.cloud.security.token.validation.Validator;

import io.vavr.control.Try;
import lombok.extern.slf4j.Slf4j;

/**
 * Workaround validator since we cannot fulfill the requirement to have HTTPS prefix on JWT issuer value.
 */
@Deprecated
@Slf4j
class CustomJwtIssuerValidator implements Validator<Token>
{
    private static final Pattern HAS_PROTOCOL_PATTERN = Pattern.compile("^\\w+://");
    private static final Predicate<String> HAS_PROTOCOL = uri -> HAS_PROTOCOL_PATTERN.matcher(uri).find();

    @Nonnull
    private final List<String> domains;

    public CustomJwtIssuerValidator( @Nonnull final OAuth2ServiceConfiguration configuration )
    {
        domains = configuration.getDomains();

        if( domains == null || domains.isEmpty() ) {
            throw new IllegalArgumentException("JwtIssuerValidator requires a domain(s).");
        }
    }

    @Override
    public ValidationResult validate( final Token token )
    {
        if( checkIssuerInDomains(token.getClaimAsString(TokenClaims.ISSUER))
            || checkIssuerInDomains(token.getClaimAsString(TokenClaims.IAS_ISSUER)) ) {
            return ValidationResults.createValid();
        }

        return ValidationResults
            .createInvalid(
                "Issuer is not trusted because it doesn't match any of these domains '{}' of the identity provider.",
                domains);
    }

    boolean checkIssuerInDomains( @Nullable final String issuer )
    {
        if( issuer == null ) {
            return false;
        }

        final String host = extractHost(issuer);
        return host != null && domains.stream().anyMatch(host::endsWith);
    }

    @Nullable
    static String extractHost( final String uri )
    {
        final String httpUri = HAS_PROTOCOL.test(uri) ? uri : "https://" + uri;
        return Try.of(() -> new URI(httpUri).getHost()).onFailure(e -> log.warn("Invalid URI: {}", uri, e)).getOrNull();
    }
}
