/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security;

import java.util.concurrent.Callable;
import java.util.concurrent.Future;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.auth0.jwt.interfaces.DecodedJWT;
import com.sap.cloud.sdk.cloudplatform.requestheader.RequestHeaderAccessor;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContext;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextAccessor;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextExecutor;
import com.sap.cloud.sdk.cloudplatform.thread.exception.ThreadContextExecutionException;
import com.sap.cloud.security.config.OAuth2ServiceConfiguration;
import com.sap.cloud.security.xsuaa.client.OAuth2TokenService;

import io.vavr.control.Try;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;

/**
 * Facade for retrieving the current {@link AuthToken}.
 */
@Slf4j
@AllArgsConstructor( access = AccessLevel.PACKAGE )
public class ScpCfAuthTokenFacade extends ExecutableAuthTokenFacade
{
    @Nonnull
    static final OAuth2TokenServiceCache DEFAULT_TOKEN_SERVICE_CACHE = OAuth2TokenServiceCache.create();

    @Nonnull
    @Deprecated
    private final OAuth2TokenServiceCache tokenServiceCache;

    @Nonnull
    private AuthTokenDecoder tokenDecoder;

    /**
     * Default constructor.
     */
    public ScpCfAuthTokenFacade()
    {
        this(DEFAULT_TOKEN_SERVICE_CACHE, createTokenDecoder());
    }

    /**
     * Create a new {@code ScpCfAuthTokenFacade} with the given custom token decoder.
     *
     * @param authTokenDecoder
     *            The {@link AuthTokenDecoder} to be used.
     */
    ScpCfAuthTokenFacade( @Nonnull final AuthTokenDecoder authTokenDecoder )
    {
        this(DEFAULT_TOKEN_SERVICE_CACHE, authTokenDecoder);
    }

    /**
     * Create a new {@code ScpCfAuthTokenFacade} with the given token service and an {@link OAuth2ServiceConfiguration}.
     * The latter will be used to construct a validator for auth token validation.
     *
     * @param tokenService
     *            The {@link OAuth2TokenService} to be used.
     * @param oauth2Configuration
     *            The OAuth2 configuration to be used.
     * @deprecated Token validation will be removed with the next major release. Please leverage the
     *             <a href="https://github.com/SAP/cloud-security-services-integration-library">security library</a>
     *             directly instead.
     */
    @Deprecated
    public ScpCfAuthTokenFacade(
        @Nonnull final OAuth2TokenService tokenService,
        @Nullable final OAuth2ServiceConfiguration oauth2Configuration )
    {
        this(OAuth2TokenServiceCache.single(tokenService), new AuthTokenDecoderXsuaa(oauth2Configuration));
    }

    @SuppressWarnings( "deprecation" )
    @Nonnull
    static AuthTokenDecoder createTokenDecoder()
    {
        final String xsuaaClassName = "com.sap.cloud.security.config.cf.CFEnvironment";
        try {
            ScpCfAuthTokenFacade.class.getClassLoader().loadClass(xsuaaClassName);
        }
        catch( final ClassNotFoundException ignored ) {
            log
                .debug(
                    "Class '{}' could not be found, assuming a version >= 3.0.0 of the XSUAA Security Library is used. AuthTokens will not be validated by the SAP Cloud SDK.",
                    xsuaaClassName);
            return new AuthTokenDecoderDefault();
        }
        log.debug("Registering xsuaa and legacy based token validators for compatibility.");
        final AuthTokenDecoder decoderXsuaa = new AuthTokenDecoderXsuaa();
        final AuthTokenDecoder decoderLegacy = new AuthTokenDecoderLegacy();
        return ( jwt ) -> Try.of(() -> decoderXsuaa.decode(jwt)).getOrElse(() -> decoderLegacy.decode(jwt));
    }

    @Nonnull
    @Override
    public Try<AuthToken> tryGetCurrentToken()
    {
        final ThreadContext currentContext = ThreadContextAccessor.getCurrentContextOrNull();
        if( currentContext != null
            && currentContext.containsProperty(AuthTokenThreadContextListener.PROPERTY_AUTH_TOKEN) ) {
            return currentContext.getPropertyValue(AuthTokenThreadContextListener.PROPERTY_AUTH_TOKEN);
        }
        return RequestHeaderAccessor.tryGetHeaderContainer().flatMap(tokenDecoder::decode);
    }

    /**
     * Attempts to retrieve an auth token for the provider account for a technical user. <strong>This method is not
     * compatible with version 3.0.0 or higher of the
     * <a href="https://github.com/SAP/cloud-security-services-integration-library">XSUAA Security Library</a> </strong>
     *
     * @return An auth token or a failure.
     * @deprecated Please leverage the
     *             <a href="https://github.com/SAP/cloud-security-services-integration-library">XSUAA Security
     *             Library</a> directly instead.
     */
    @Nonnull
    @Deprecated
    public Try<AuthToken> tryGetXsuaaServiceToken()
    {
        return Try.of(() -> new AuthTokenRequest(tokenDecoder, tokenServiceCache).getXsuaaServiceToken());
    }

    /**
     * Asynchronously resolve a new refresh token on behalf of a given JWT.
     *
     * @param jwt
     *            The JWT.
     * @return A new refresh token.
     * @deprecated Please leverage the
     *             <a href="https://github.com/SAP/cloud-security-services-integration-library">security library</a>
     *             directly instead.
     */
    @Nonnull
    @Deprecated
    public Future<String> getRefreshToken( @Nonnull final DecodedJWT jwt )
    {
        AuthTokenDecoder decoder = tokenDecoder;
        if( !(decoder instanceof AuthTokenDecoderXsuaa) ) {
            decoder = new AuthTokenDecoderXsuaa();
        }
        return ((AuthTokenDecoderXsuaa) decoder).getRefreshToken(jwt, tokenServiceCache);
    }

    @Nullable
    @Override
    protected <T> T executeWithAuthToken( @Nonnull final AuthToken authToken, @Nonnull final Callable<T> callable )
        throws ThreadContextExecutionException
    {
        return ThreadContextExecutor
            .fromCurrentOrNewContext()
            .withListeners(new AuthTokenThreadContextListener(authToken))
            .execute(callable);
    }
}
