/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.servlet.response;

import javax.annotation.Nonnull;

import com.sap.cloud.sdk.cloudplatform.servlet.response.mapper.ResponseMapper;

import lombok.Getter;

/**
 * Base class to be used by different the exception handling of different frameworks.
 * <p>
 * Provides different methods to modify the underlying {@code Exception} mapping.
 * 
 * @param <ClassT>
 *            The type of the implementing class.
 */
public abstract class AbstractExceptionMapper<ClassT extends AbstractExceptionMapper<ClassT>>
{
    @Getter
    @Nonnull
    private final ErrorResponseBuilder errorResponseBuilder = ErrorResponseBuilder.newBuilder();

    /**
     * Getter for {@code this}. Needed to facilitate a fluent interface in the implementing classes.
     * 
     * @return this
     */
    @Nonnull
    protected abstract ClassT self();

    /**
     * Wraps the given {@code Throwable} into a {@code ResponseWithErrorCode}, using the mappings stored in the
     * {@code ErrorResponseBuilder}.
     * 
     * @param throwable
     *            The {@code Throwable} to handle.
     * @return The {@code ResponseWithErrorCode} for the given {@code Throwable}.
     */
    @Nonnull
    protected ResponseWithErrorCode toResponseWithErrorCode( @Nonnull final Throwable throwable )
    {
        return getErrorResponseBuilder().build(throwable);
    }

    /**
     * Removes the mapping for the given {@code Throwable} class.
     * 
     * @param tClass
     *            The class to remove the mapping for.
     * @param <ThrowableT>
     *            The type of the class to be removed.
     * @return this
     */
    @Nonnull
    public <ThrowableT extends Throwable> ClassT removeMapping( //ALLOW THROWABLE
        @Nonnull final Class<ThrowableT> tClass )
    {
        getErrorResponseBuilder().removeMapping(tClass);
        return self();
    }

    /**
     * Adds the given {@code ResponseMapper} for the {@code Throwable} specified insider the mapper.
     * 
     * @param mapper
     *            The mapper to add.
     * @param <ThrowableT>
     *            The type of the {@code Throwable} the mapper handles.
     * @return this
     */
    @Nonnull
    public <ThrowableT extends Throwable> ClassT withMapper( //ALLOW THROWABLE
        @Nonnull final ResponseMapper<ThrowableT> mapper )
    {
        getErrorResponseBuilder().withMapper(mapper);
        return self();
    }

    /**
     * Adds the given {@code ResponseMapper} for the {@code Throwable} specified insider the mapper. Also adds a custom
     * {@code LogLevel} used to server-side log any {@code Throwable} of the specified type.
     * 
     * @param responseMapper
     *            The mapper to add.
     * @param logLevel
     *            The {@code LogLevel} to use for server-side logging of the {@code Throwable}.
     * @param <ThrowableT>
     *            The type of the exception to be handled.
     * @return this
     */
    @Nonnull
    public <ThrowableT extends Throwable & WithErrorResponse> ClassT withMapper( //ALLOW THROWABLE
        @Nonnull final ResponseMapper<ThrowableT> responseMapper,
        @Nonnull final LogLevel logLevel )
    {
        getErrorResponseBuilder().withMapper(responseMapper, logLevel);
        return self();
    }

    /**
     * Only sets a specific server-side log level for any {@code Throwable} of the given type.
     * 
     * @param throwableClass
     *            The class of {@code Throwable} which should be logged with a specific {@code LogLevel}.
     * @param logLevel
     *            The {@code LogLevel} to use.
     * @param <ThrowableT>
     *            The type of the {@code Throwable} to log.
     * @return this
     */
    @Nonnull
    public <ThrowableT extends Throwable & WithErrorResponse> ClassT logAsLevel( //ALLOW THROWABLE
        @Nonnull final Class<ThrowableT> throwableClass,
        @Nonnull final LogLevel logLevel )
    {
        getErrorResponseBuilder().logAsLevel(throwableClass, logLevel);
        return self();
    }

    /**
     * Only sets the error log level for any {@code Throwable} of the given type.
     * 
     * @param throwableClass
     *            The class of {@code Throwable} which should be logged as error.
     * @param <ThrowableT>
     *            The type of the {@code Throwable} to log.
     * @return this
     */
    @Nonnull
    public <ThrowableT extends Throwable & WithErrorResponse> //ALLOW THROWABLE
    ClassT logAsError( @Nonnull final Class<ThrowableT> throwableClass )
    {
        getErrorResponseBuilder().logAsError(throwableClass);
        return self();
    }

    /**
     * Only sets the warning log level for any {@code Throwable} of the given type.
     * 
     * @param throwableClass
     *            The class of {@code Throwable} which should be logged as warning.
     * @param <ThrowableT>
     *            The type of the {@code Throwable} to log.
     * @return this
     */
    @Nonnull
    public <ThrowableT extends Throwable & WithErrorResponse> ClassT logAsWarning( //ALLOW THROWABLE
        @Nonnull final Class<ThrowableT> throwableClass )
    {
        getErrorResponseBuilder().logAsWarning(throwableClass);
        return self();
    }

    /**
     * Only sets the info log level for any {@code Throwable} of the given type.
     * 
     * @param throwableClass
     *            The class of {@code Throwable} which should be logged as info.
     * @param <ThrowableT>
     *            The type of the {@code Throwable} to log.
     * @return this
     */
    @Nonnull
    public <ThrowableT extends Throwable & WithErrorResponse> ClassT logAsInfo( //ALLOW THROWABLE
        @Nonnull final Class<ThrowableT> throwableClass )
    {
        getErrorResponseBuilder().logAsInfo(throwableClass);
        return self();
    }
}
