/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.servlet;

import java.util.concurrent.Callable;
import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;

import com.sap.cloud.sdk.cloudplatform.servlet.exception.RequestAccessException;
import com.sap.cloud.sdk.cloudplatform.thread.DefaultThreadContext;
import com.sap.cloud.sdk.cloudplatform.thread.Executable;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextExecutor;
import com.sap.cloud.sdk.cloudplatform.thread.exception.ThreadContextExecutionException;

import io.vavr.control.Try;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * Accessor for retrieving the current {@link HttpServletRequest}.
 */
@NoArgsConstructor( access = AccessLevel.PRIVATE )
@Slf4j
public final class RequestAccessor
{
    /**
     * The {@link RequestFacade} instance.
     */
    @Getter
    @Nonnull
    private static RequestFacade requestFacade = new DefaultRequestFacade();

    /**
     * Global fallback {@link HttpServletRequest}. By default, no fallback is used, i.e., the fallback is {@code null}.
     * A global fallback can be useful to ease testing with a mocked request.
     */
    @Getter
    @Setter
    @Nullable
    private static Supplier<HttpServletRequest> fallbackRequest = null;

    /**
     * Replaces the default {@link RequestFacade} instance.
     *
     * @param requestFacade
     *            An instance of {@link RequestFacade}. Use {@code null} to reset the facade.
     */
    public static void setRequestFacade( @Nullable final RequestFacade requestFacade )
    {
        if( requestFacade == null ) {
            RequestAccessor.requestFacade = new DefaultRequestFacade();
        } else {
            RequestAccessor.requestFacade = requestFacade;
        }
    }

    /**
     * Returns the current {@link HttpServletRequest}, delegating to {@link #tryGetCurrentRequest()} and unwrapping the
     * {@link Try}.
     *
     * @return The current {@link HttpServletRequest}.
     *
     * @throws RequestAccessException
     *             If there is an issue while accessing the {@link HttpServletRequest}.
     */
    @Nonnull
    public static HttpServletRequest getCurrentRequest()
        throws RequestAccessException
    {
        return tryGetCurrentRequest().getOrElseThrow(failure -> {
            if( failure instanceof RequestAccessException ) {
                throw (RequestAccessException) failure;
            } else {
                throw new RequestAccessException("Failed to get current request.", failure);
            }
        });
    }

    /**
     * Returns a {@link Try} of the current {@link HttpServletRequest}, or, if the {@link Try} is a failure, the global
     * fallback.
     *
     * @return A {@link Try} of the current {@link HttpServletRequest}.
     */
    @Nonnull
    public static Try<HttpServletRequest> tryGetCurrentRequest()
    {
        final Try<HttpServletRequest> requestTry = requestFacade.tryGetCurrentRequest();

        if( requestTry.isFailure() && fallbackRequest != null ) {
            return requestTry.recover(failure -> {
                final HttpServletRequest fallback = fallbackRequest.get();
                log.warn("Recovering with fallback request: {}.", fallback, failure);
                return fallback;
            });
        }

        return requestTry;
    }

    /**
     * Execute the given {@link Callable} with a given request.
     *
     * @param request
     *            The request to execute with.
     * @param callable
     *            The callable to execute.
     *
     * @param <T>
     *            The type of the callable.
     *
     * @return The value computed by the callable.
     *
     * @throws ThreadContextExecutionException
     *             If there is an issue while running the code with the request.
     */
    @Nullable
    public static <
        T> T executeWithRequest( @Nonnull final HttpServletRequest request, @Nonnull final Callable<T> callable )
            throws ThreadContextExecutionException
    {
        return new ThreadContextExecutor()
            .withThreadContext(new DefaultThreadContext())
            .withListeners(new RequestThreadContextListener(request))
            .execute(callable);
    }

    /**
     * Execute the given {@link Executable} with a given request.
     *
     * @param request
     *            The request to execute with.
     * @param executable
     *            The executable to execute.
     *
     * @throws ThreadContextExecutionException
     *             If there is an issue while running the code with the request.
     */
    public static
        void
        executeWithRequest( @Nonnull final HttpServletRequest request, @Nonnull final Executable executable )
            throws ThreadContextExecutionException
    {
        executeWithRequest(request, () -> {
            executable.execute();
            return null;
        });
    }

    /**
     * Execute the given {@link Callable}, using the given request as fallback if there is no other request available.
     *
     * @param fallbackRequest
     *            The request to fall back to.
     * @param callable
     *            The callable to execute.
     *
     * @param <T>
     *            The type of the callable.
     *
     * @return The value computed by the callable.
     *
     * @throws ThreadContextExecutionException
     *             If there is an issue while running the code with the request.
     */
    @Nullable
    public static <T> T executeWithFallbackRequest(
        @Nonnull final Supplier<HttpServletRequest> fallbackRequest,
        @Nonnull final Callable<T> callable )
        throws ThreadContextExecutionException
    {
        final Try<HttpServletRequest> requestTry = tryGetCurrentRequest();

        if( requestTry.isSuccess() ) {
            try {
                return callable.call();
            }
            catch( final ThreadContextExecutionException e ) {
                throw e;
            }
            catch( final Exception e ) {
                throw new ThreadContextExecutionException(e);
            }
        }

        return executeWithRequest(fallbackRequest.get(), callable);
    }

    /**
     * Execute the given {@link Executable}, using the given request as fallback if there is no other request available.
     *
     * @param fallbackRequest
     *            The request to fall back to.
     * @param executable
     *            The executable to execute.
     *
     * @throws ThreadContextExecutionException
     *             If there is an issue while running the code with the request.
     */
    public static void executeWithFallbackRequest(
        @Nonnull final Supplier<HttpServletRequest> fallbackRequest,
        @Nonnull final Executable executable )
        throws ThreadContextExecutionException
    {
        executeWithFallbackRequest(fallbackRequest, () -> {
            executable.execute();
            return null;
        });
    }
}
