/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.servlet.response;

import java.util.UUID;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.gson.annotations.Expose;
import com.google.gson.annotations.SerializedName;

import io.vavr.control.Option;
import lombok.Getter;

/**
 * Class representing an error response that is returned by a web service.
 */
@JsonInclude( JsonInclude.Include.NON_NULL )
public abstract class ResponseWithErrorCode
{
    /**
     * The HTTP status code.
     */
    @Getter
    @JsonIgnore
    protected final int httpStatusCode;

    /**
     * The error code.
     */
    @Getter
    @Expose
    @SerializedName( "error" )
    @JsonProperty( "error" )
    protected final String errorCode;

    /**
     * The error reference id.
     */
    @Getter
    @Expose
    @SerializedName( "referenceId" )
    @JsonProperty( "referenceId" )
    protected final String referenceId;

    /**
     * The error description.
     */
    @Nullable
    @Expose
    @SerializedName( "description" )
    @JsonProperty( "description" )
    protected final String description;

    /**
     * Constructor for a response with a HTTP status code.
     * 
     * @param httpStatusCode
     *            The HTTP status code.
     * @param errorCode
     *            The error code.
     * @param description
     *            The optional error description.
     */
    public ResponseWithErrorCode(
        final int httpStatusCode,
        @Nonnull final String errorCode,
        @Nullable final String description )
    {
        this.httpStatusCode = httpStatusCode;
        this.errorCode = errorCode;
        this.description = description;

        referenceId = UUID.randomUUID().toString();
    }

    /**
     * Get optional description.
     * 
     * @return The description.
     */
    @JsonIgnore
    @Nonnull
    public Option<String> getDescription()
    {
        return Option.of(description);
    }
}
