/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.servlet;

import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;

import com.sap.cloud.sdk.cloudplatform.requestheader.DefaultRequestHeaderContainer;
import com.sap.cloud.sdk.cloudplatform.requestheader.RequestHeaderContainer;
import com.sap.cloud.sdk.cloudplatform.requestheader.RequestHeaderThreadContextListener;
import com.sap.cloud.sdk.cloudplatform.thread.Property;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContext;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextListener;

import io.vavr.control.Option;
import io.vavr.control.Try;
import lombok.RequiredArgsConstructor;

/**
 * Implementation of {@link ThreadContextListener} that ensures that the current
 * {@link javax.servlet.http.HttpServletRequest}
 */
@RequiredArgsConstructor
public class RequestThreadContextListener implements ThreadContextListener
{
    /**
     * The ThreadContext key.
     */
    public static final String PROPERTY_REQUEST = RequestThreadContextListener.class.getName() + ":request";

    /**
     * The {@link HttpServletRequest} to be used by this listener.
     */
    @Nonnull
    private final HttpServletRequest request;

    @Override
    public int getPriority()
    {
        return DefaultPriorities.REQUEST_LISTENER;
    }

    @Override
    public void beforeInitialize(
        @Nonnull final ThreadContext threadContext,
        @Nullable final ThreadContext parentThreadContext )
    {
        threadContext.setPropertyIfAbsent(PROPERTY_REQUEST, Try.of(() -> Property.of(request)));
        threadContext.setPropertyIfAbsent(
            RequestHeaderThreadContextListener.PROPERTY_REQUEST_HEADERS,
            Try.of(() -> Property.of(extractHeaders(request))));
    }

    @Override
    public
        void
        beforeDestroy( @Nonnull final ThreadContext threadContext, @Nullable final ThreadContext parentThreadContext )
    {
        threadContext.removeProperty(PROPERTY_REQUEST);

        if( threadContext.containsProperty(RequestHeaderThreadContextListener.PROPERTY_REQUEST_HEADERS) ) {
            // make sure we do not remove the properties twice after "RequestThreadContextListener" was already invoked.
            threadContext.removeProperty(RequestHeaderThreadContextListener.PROPERTY_REQUEST_HEADERS);
        }
    }

    @Nonnull
    private static RequestHeaderContainer extractHeaders( @Nonnull final HttpServletRequest request )
    {
        final Enumeration<String> headerNames = request.getHeaderNames();
        if( headerNames == null ) {
            return RequestHeaderContainer.EMPTY;
        }

        final Map<String, Collection<String>> headers = new HashMap<>();
        while( headerNames.hasMoreElements() ) {
            final Option<String> headerName = Option.of(headerNames.nextElement());
            headerName.map(request::getHeaders).filter(Objects::nonNull).peek(
                values -> headers.put(headerName.get(), Collections.list(values)));
        }
        return DefaultRequestHeaderContainer.fromMultiValueMap(headers);
    }
}
