/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.servlet.response;

import java.util.IdentityHashMap;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.cloudplatform.servlet.exception.InvalidParameterException;
import com.sap.cloud.sdk.cloudplatform.servlet.exception.PayloadTooLargeException;
import com.sap.cloud.sdk.cloudplatform.servlet.response.mapper.AbstractResponseMapper;
import com.sap.cloud.sdk.cloudplatform.servlet.response.mapper.ConstraintViolationExceptionMapper;
import com.sap.cloud.sdk.cloudplatform.servlet.response.mapper.EntityAlreadyExistsExceptionMapper;
import com.sap.cloud.sdk.cloudplatform.servlet.response.mapper.EntityNotFoundExceptionMapper;
import com.sap.cloud.sdk.cloudplatform.servlet.response.mapper.IllegalArgumentExceptionMapper;
import com.sap.cloud.sdk.cloudplatform.servlet.response.mapper.JsonSyntaxExceptionMapper;
import com.sap.cloud.sdk.cloudplatform.servlet.response.mapper.NumberFormatExceptionMapper;
import com.sap.cloud.sdk.cloudplatform.servlet.response.mapper.PropertyBindingExceptionMapper;
import com.sap.cloud.sdk.cloudplatform.servlet.response.mapper.ResponseMapper;
import com.sap.cloud.sdk.cloudplatform.servlet.response.mapper.StringParsingExceptionMapper;
import com.sap.cloud.sdk.cloudplatform.servlet.response.mapper.TimeoutExceptionMapper;
import com.sap.cloud.sdk.cloudplatform.servlet.response.mapper.UnsupportedOperationExceptionMapper;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

/**
 * Builder translating {@link Throwable}s to {@link ResponseWithErrorCode}s.
 */
@SuppressWarnings( "PMD.UseOfNonJmxLogger" )
@Slf4j
public class ErrorResponseBuilder
{
    /**
     * The maximum stack trace level to recurse into during error response building.
     */
    public static final int MAX_STACK_TRACE_LEVEL = 20;

    /**
     * The response mappers by class.
     */
    protected final IdentityHashMap<Class<?>, ResponseMapper<?>> responseMappers = Maps.newIdentityHashMap();

    /**
     * Creates a new default error response builder.
     *
     * @return The new builder.
     */
    @Nonnull
    public static ErrorResponseBuilder newBuilder()
    {
        return new ErrorResponseBuilder()
            .withMapper(new IllegalArgumentExceptionMapper())
            .withMapper(new JsonSyntaxExceptionMapper())
            .withMapper(new NumberFormatExceptionMapper())
            .withMapper(new PropertyBindingExceptionMapper())
            .withMapper(new StringParsingExceptionMapper())
            .withMapper(new TimeoutExceptionMapper())
            .withMapper(new UnsupportedOperationExceptionMapper())
            .withMapper(new ConstraintViolationExceptionMapper())
            .withMapper(new EntityAlreadyExistsExceptionMapper())
            .withMapper(new EntityNotFoundExceptionMapper())
            .logAsWarning(InvalidParameterException.class)
            .logAsWarning(PayloadTooLargeException.class);
    }

    @RequiredArgsConstructor
    private static class LogLevelDelegateMapper<ThrowableT extends Throwable & WithErrorResponse> //ALLOW THROWABLE
        extends
        AbstractResponseMapper<ThrowableT>
    {
        @Nonnull
        private final ResponseMapper<ThrowableT> delegate;

        @Nonnull
        private final LogLevel logLevel;

        @Nonnull
        @Override
        public Class<ThrowableT> getThrowableClass()
        {
            return delegate.getThrowableClass();
        }

        @Nullable
        @Override
        public ResponseWithErrorCode toResponse( @Nonnull final Throwable throwable )
        {
            return delegate.toResponse(throwable);
        }

        @Nonnull
        @Override
        public LogLevel getLogLevel( @Nonnull final Throwable throwable )
        {
            return logLevel;
        }

        @Nullable
        @Override
        public String getErrorMessage( @Nonnull final Throwable throwable )
        {
            return delegate.getErrorMessage(throwable);
        }
    }

    /**
     * Use the given {@link ResponseMapper} for building the error response.
     *
     * @param responseMapper
     *            The response mapper to use.
     *
     * @return The builder for the error response.
     */
    @Nonnull
    public ErrorResponseBuilder withMapper( @Nonnull final ResponseMapper<?> responseMapper )
    {
        final Class<?> throwableClass = responseMapper.getThrowableClass();

        responseMappers.put(throwableClass, responseMapper);

        if( log.isInfoEnabled() ) {
            log.info("Successfully set {} for {}.", responseMapper.getClass().getName(), throwableClass.getName());
        }

        return this;
    }

    /**
     * Use the given {@link ResponseMapper} for building the error response.
     *
     * @param responseMapper
     *            The response mapper to use.
     * @param logLevel
     *            The log level to use with the given response mapper.
     * @param <ThrowableT>
     *            The generic throwable type.
     *
     * @return The builder for the error response.
     */
    @Nonnull
    public <ThrowableT extends Throwable & WithErrorResponse> ErrorResponseBuilder withMapper( //ALLOW THROWABLE
        @Nonnull final ResponseMapper<ThrowableT> responseMapper,
        @Nonnull final LogLevel logLevel )
    {
        final Class<ThrowableT> throwableClass = responseMapper.getThrowableClass();

        responseMappers.put(throwableClass, new LogLevelDelegateMapper<>(responseMapper, logLevel));

        if( log.isInfoEnabled() ) {
            log.info(
                "Successfully set {} for {} with log level {}.",
                responseMapper.getClass().getName(),
                throwableClass.getName(),
                logLevel);
        }

        return this;
    }

    /**
     * Defines the log level to use for the given type of Throwable.
     *
     * @param throwableClass
     *            The Throwable class.
     * @param logLevel
     *            The log level to use.
     * @param <ThrowableT>
     *            The generic throwable type.
     *
     * @return The builder for the error response.
     */
    @Nonnull
    public <ThrowableT extends Throwable & WithErrorResponse> ErrorResponseBuilder logAsLevel( //ALLOW THROWABLE
        @Nonnull final Class<ThrowableT> throwableClass,
        @Nonnull final LogLevel logLevel )
    {
        final ResponseMapper<?> responseMapper = responseMappers.get(throwableClass);

        if( responseMapper != null ) {
            @SuppressWarnings( "unchecked" )
            final ResponseMapper<ThrowableT> mapper = (ResponseMapper<ThrowableT>) responseMapper;

            return withMapper(mapper, logLevel);
        } else {
            responseMappers.put(throwableClass, new AbstractResponseMapper<ThrowableT>()
            {
                @Nonnull
                @Override
                public Class<ThrowableT> getThrowableClass()
                {
                    return throwableClass;
                }

                @Nonnull
                @Override
                public ResponseWithErrorCode toResponse( @Nonnull final Throwable throwable )
                {
                    return ((WithErrorResponse) throwable).getErrorResponse();
                }

                @Nonnull
                @Override
                public LogLevel getLogLevel( @Nonnull final Throwable throwable )
                {
                    return logLevel;
                }
            });

            if( log.isInfoEnabled() ) {
                log.info("Using log level {} for {}.", logLevel, throwableClass.getName());
            }

            return this;
        }
    }

    /**
     * Defines the use of log level "error" for the given type of Throwable.
     *
     * @param throwableClass
     *            The Throwable class.
     * @param <ThrowableT>
     *            The generic throwable type.
     *
     * @return The builder for the error response.
     */
    @Nonnull
    public <ThrowableT extends Throwable & WithErrorResponse> ErrorResponseBuilder logAsError( //ALLOW THROWABLE
        @Nonnull final Class<ThrowableT> throwableClass )
    {
        return logAsLevel(throwableClass, LogLevel.INFO);
    }

    /**
     * Defines the use of log level "warning" for the given type of Throwable.
     *
     * @param throwableClass
     *            The Throwable class.
     * @param <ThrowableT>
     *            The generic throwable type.
     *
     * @return The builder for the error response.
     */
    @Nonnull
    public <ThrowableT extends Throwable & WithErrorResponse> ErrorResponseBuilder logAsWarning( //ALLOW THROWABLE
        @Nonnull final Class<ThrowableT> throwableClass )
    {
        return logAsLevel(throwableClass, LogLevel.WARNING);
    }

    /**
     * Defines the use of log level "info" for the given type of Throwable.
     *
     * @param throwableClass
     *            The Throwable class.
     * @param <ThrowableT>
     *            The generic throwable type.
     *
     * @return The builder for the error response.
     */
    @Nonnull
    public <ThrowableT extends Throwable & WithErrorResponse> ErrorResponseBuilder logAsInfo( //ALLOW THROWABLE
        @Nonnull final Class<ThrowableT> throwableClass )
    {
        return logAsLevel(throwableClass, LogLevel.INFO);
    }

    /**
     * Removes the mapping for a given type of Throwable.
     *
     * @param throwableClass
     *            The Throwable class.
     * @param <ThrowableT>
     *            The generic throwable type.
     *
     * @return The builder for the error response.
     */
    @Nonnull
    public <ThrowableT extends Throwable> ErrorResponseBuilder removeMapping( //ALLOW THROWABLE
        @Nonnull final Class<ThrowableT> throwableClass )
    {
        final ResponseMapper<?> removed = responseMappers.remove(throwableClass);
        if( log.isInfoEnabled() ) {
            if( removed != null ) {
                log.info("Successfully removed {} for {}.", removed.getClass().getName(), throwableClass.getName());
            } else {
                log.info("There was no ResponseMapper for {} that could be removed.", throwableClass.getName());
            }
        }
        return this;
    }

    /**
     * Create the response instance for a given error.
     * 
     * @param throwable
     *            The throwable reference.
     * @return The new response created by the builder.
     */
    @Nonnull
    public ResponseWithErrorCode build( @Nonnull final Throwable throwable )
    {
        return toResponse(throwable, 0);
    }

    /**
     * Get maximum stack trace level.
     * 
     * @return The maximum stack trace level.
     */
    protected int getMaxStackTraceLevel()
    {
        return MAX_STACK_TRACE_LEVEL;
    }

    /**
     * Translate the builder information to a response instance.
     * 
     * @param throwable
     *            The throwable reference.
     * @param stackTraceLevel
     *            The stack trace level.
     * @return The new response created by the builder.
     */
    protected ResponseWithErrorCode toResponse( @Nonnull final Throwable throwable, final int stackTraceLevel )
    {
        if( stackTraceLevel >= getMaxStackTraceLevel() ) {
            return toResponse(throwable);
        }

        if( throwable.getCause() != null && throwable != throwable.getCause() ) {
            toResponse(throwable.getCause(), stackTraceLevel + 1);
        }

        return toResponse(throwable);
    }

    /**
     * Translate the builder information to a response instance.
     * 
     * @param throwable
     *            The throwable reference.
     * @return The new response created by the builder.
     */
    protected ResponseWithErrorCode toResponse( @Nonnull final Throwable throwable )
    {
        final ResponseMapper<?> responseMapper = responseMappers.get(throwable.getClass());

        ResponseWithErrorCode response = null;
        LogLevel logLevel = null;
        String logMessage = null;

        if( responseMapper != null ) {
            response = responseMapper.toResponse(throwable);
            logLevel = responseMapper.getLogLevel(throwable);
            logMessage = responseMapper.getErrorMessage(throwable);
        }

        if( response == null ) {
            if( throwable instanceof WithErrorResponse ) {
                response = ((WithErrorResponse) throwable).getErrorResponse();
            } else {
                response =
                    new UnexpectedErrorResponse(
                        "An unexpected error occurred. "
                            + "For details, please refer to the application log entry with the given referenceId.");
            }
        }

        if( logLevel == null ) {
            logLevel = LogLevel.ERROR;
        }

        if( logMessage == null ) {
            logMessage = throwable.getMessage();
        }

        logMessage =
            "Handling throwable with error response (referenceId: "
                + response.getReferenceId()
                + ")."
                + (logMessage == null ? "" : " Message: " + logMessage);

        if( logLevel == LogLevel.WARNING ) {
            log.warn(logMessage, throwable);
        } else if( logLevel == LogLevel.INFO ) {
            log.info(logMessage, throwable);
        } else {
            log.error(logMessage, throwable);
        }

        return response;
    }
}
