/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.tenant;

import java.net.URI;

import javax.annotation.Nonnull;

import com.auth0.jwt.interfaces.DecodedJWT;
import com.auth0.jwt.interfaces.Payload;
import com.sap.cloud.sdk.cloudplatform.security.AuthToken;
import com.sap.cloud.sdk.cloudplatform.security.AuthTokenAccessor;
import com.sap.cloud.sdk.cloudplatform.tenant.exception.TenantAccessException;

import io.vavr.control.Try;
import lombok.extern.slf4j.Slf4j;

/**
 * Facade providing access to tenant information on SAP Cloud Platform Cloud Foundry.
 */
@Slf4j
public class ScpCfTenantFacade extends DefaultTenantFacade
{
    private static final String JWT_ZONE_ID = "zid";
    private static final String JWT_ISSUER = "iss";

    @Nonnull
    private String getTenantId( @Nonnull final Payload jwt )
    {
        final String tenantId = jwt.getClaim(JWT_ZONE_ID).asString();

        if( tenantId == null ) {
            throw new TenantAccessException("No tenant identifier ('" + JWT_ZONE_ID + "' element) found in JWT.");
        }

        return tenantId;
    }

    @Nonnull
    private String getIssuer( @Nonnull final DecodedJWT jwt )
    {
        final String issuer = jwt.getClaim(JWT_ISSUER).asString();

        if( issuer == null ) {
            throw new TenantAccessException("No subdomain ('" + JWT_ISSUER + "' element) found in JWT.");
        }

        return issuer;
    }

    @Nonnull
    private String getSubdomain( @Nonnull final URI issuerUri )
    {
        final String host = issuerUri.getHost();

        if( host == null || !host.contains(".") ) {
            throw new TenantAccessException("Failed to get subdomain from issuer URI '" + issuerUri + "'.");
        }

        return host.split("\\.")[0];
    }

    @Nonnull
    private Try<Tenant> tryGetTenant( @Nonnull final Try<AuthToken> authToken )
    {
        return super.tryGetCurrentTenant().orElse(() -> {
            final Try<DecodedJWT> jwtTry = authToken.map(AuthToken::getJwt);

            if( jwtTry.isFailure() ) {
                return Try.failure(jwtTry.getCause());
            }

            final Try<String> tenantIdTry = jwtTry.map(this::getTenantId);

            if( tenantIdTry.isFailure() ) {
                return Try.failure(tenantIdTry.getCause());
            }

            final Try<String> subdomainTry = jwtTry.map(this::getIssuer).map(URI::create).map(this::getSubdomain);

            if( subdomainTry.isFailure() ) {
                return Try.failure(subdomainTry.getCause());
            }

            return Try.of(() -> new ScpCfTenant(tenantIdTry.get(), subdomainTry.get()));
        });
    }

    /**
     * Returns a {@link Try} of the {@link Tenant} that may be contained in the current {@link AuthToken}.
     *
     * @return A {@link Try} of the {@link Tenant}.
     */
    @Nonnull
    public Try<Tenant> tryGetAuthTokenTenant()
    {
        return super.tryGetCurrentTenant().orElse(() -> tryGetTenant(AuthTokenAccessor.tryGetCurrentToken()));
    }

    /**
     * Returns a {@link Try} of the {@link Tenant} that may be contained in the current {@link AuthToken}.
     *
     * @return A {@link Try} of the {@link Tenant}.
     */
    @Nonnull
    public Try<Tenant> tryGetXsuaaServiceTenant()
    {
        return super.tryGetCurrentTenant().orElse(() -> tryGetTenant(AuthTokenAccessor.tryGetXsuaaServiceToken()));
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public Try<Tenant> tryGetCurrentTenant()
    {
        return super.tryGetCurrentTenant().orElse(
            () -> tryGetTenant(
                AuthTokenAccessor.tryGetCurrentToken().orElse(AuthTokenAccessor::tryGetXsuaaServiceToken)));
    }
}
