/*
 * Copyright (c) 2023 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.tenant;

import java.net.URI;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nonnull;

import com.auth0.jwt.interfaces.DecodedJWT;
import com.auth0.jwt.interfaces.Payload;
import com.google.common.collect.Iterables;
import com.google.common.collect.Streams;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.exception.CloudPlatformException;
import com.sap.cloud.sdk.cloudplatform.security.AuthToken;
import com.sap.cloud.sdk.cloudplatform.security.AuthTokenAccessor;
import com.sap.cloud.sdk.cloudplatform.security.AuthTokenFacade;
import com.sap.cloud.sdk.cloudplatform.security.ScpCfAuthTokenFacade;
import com.sap.cloud.sdk.cloudplatform.security.exception.AuthTokenAccessException;
import com.sap.cloud.sdk.cloudplatform.tenant.exception.TenantAccessException;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContext;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextAccessor;

import io.vavr.control.Try;
import lombok.extern.slf4j.Slf4j;

/**
 * Facade providing access to tenant information on SAP Business Technology Platform Cloud Foundry.
 */
@Slf4j
public class ScpCfTenantFacade extends DefaultTenantFacade
{
    private static final String XSUAA_JWT_ZONE_ID = "zid";
    private static final String IAS_JWT_ZONE_ID = "zone_uuid";
    private static final String IAS_JWT_APP_TID = "app_tid";
    private static final List<String> TENANT_ID_CLAIMS =
        Arrays.asList(XSUAA_JWT_ZONE_ID, IAS_JWT_APP_TID, IAS_JWT_ZONE_ID);
    private static final String JWT_ISSUER = "iss";

    @Nonnull
    private String getTenantId( @Nonnull final Payload jwt )
    {
        final Optional<String> maybeTenantId =
            TENANT_ID_CLAIMS.stream().map(id -> jwt.getClaim(id).asString()).filter(Objects::nonNull).findFirst();

        if( !maybeTenantId.isPresent() ) {
            throw new TenantAccessException(
                "No tenant/zone identifier (one of these elements [" + TENANT_ID_CLAIMS + "]) found in JWT.");
        }

        return maybeTenantId.get();
    }

    @Nonnull
    private String getIssuer( @Nonnull final DecodedJWT jwt )
    {
        final String issuer = jwt.getClaim(JWT_ISSUER).asString();

        if( issuer == null ) {
            throw new TenantAccessException("No subdomain ('" + JWT_ISSUER + "' element) found in JWT.");
        }

        return issuer;
    }

    @Nonnull
    private String getSubdomain( @Nonnull final URI issuerUri )
    {
        final String host = issuerUri.getHost();

        if( host == null || !host.contains(".") ) {
            throw new TenantAccessException("Failed to get subdomain from issuer URI '" + issuerUri + "'.");
        }

        return host.split("\\.")[0];
    }

    @Nonnull
    private Try<Tenant> tryGetTenantFromAuthToken( @Nonnull final Try<AuthToken> authToken )
    {
        final Try<DecodedJWT> jwtTry = authToken.map(AuthToken::getJwt);

        if( jwtTry.isFailure() ) {
            return Try.failure(jwtTry.getCause());
        }

        final Try<String> tenantIdTry = jwtTry.map(this::getTenantId);

        if( tenantIdTry.isFailure() ) {
            return Try.failure(tenantIdTry.getCause());
        }

        final Try<String> subdomainTry = jwtTry.map(this::getIssuer).map(URI::create).map(this::getSubdomain);

        if( subdomainTry.isFailure() ) {
            return Try.failure(subdomainTry.getCause());
        }

        return Try.of(() -> new ScpCfTenant(tenantIdTry.get(), subdomainTry.get()));
    }

    /**
     * Get the {@link Tenant} that may be contained in the current {@link AuthToken}.
     *
     * @return A {@link Try} of the {@link Tenant}.
     */
    @Nonnull
    public Try<Tenant> tryGetAuthTokenTenant()
    {
        return super.tryGetCurrentTenant()
            .orElse(() -> tryGetTenantFromAuthToken(AuthTokenAccessor.tryGetCurrentToken()));
    }

    /**
     * Get the {@link Tenant} that may be contained in the current {@link AuthToken}. <strong>This method is not
     * compatible with version 3.0.0 or higher of the
     * <a href="https://github.com/SAP/cloud-security-services-integration-library">XSUAA Security Library</a> </strong>
     *
     * @return A {@link Try} of the {@link Tenant}.
     * @deprecated Please leverage the
     *             <a href="https://github.com/SAP/cloud-security-services-integration-library">security library</a>
     *             directly instead.
     */
    @Nonnull
    @Deprecated
    public Try<Tenant> tryGetXsuaaServiceTenant()
    {
        return super.tryGetCurrentTenant().orElse(() -> tryGetTenantFromAuthToken(tryGetXsuaaServiceToken()));
    }

    @SuppressWarnings( "deprecation" )
    @Nonnull
    @Override
    public Try<Tenant> tryGetCurrentTenant()
    {
        final ThreadContext currentContext = ThreadContextAccessor.getCurrentContextOrNull();
        if( currentContext != null && currentContext.containsProperty(TenantThreadContextListener.PROPERTY_TENANT) ) {
            return currentContext.getPropertyValue(TenantThreadContextListener.PROPERTY_TENANT);
        }
        return tryGetTenantFromAuthToken(AuthTokenAccessor.tryGetCurrentToken()) // read from user token
            .orElse(
                () -> tryGetTenantFromServiceBinding(
                    com.sap.cloud.sdk.cloudplatform.CloudPlatformAccessor.tryGetCloudPlatform())); // read bindings
    }

    @Deprecated
    private Try<AuthToken> tryGetXsuaaServiceToken()
    {
        final AuthTokenFacade authTokenFacade = AuthTokenAccessor.getAuthTokenFacade();
        if( authTokenFacade instanceof ScpCfAuthTokenFacade ) {
            return ((ScpCfAuthTokenFacade) authTokenFacade).tryGetXsuaaServiceToken();
        }
        return Try
            .failure(
                new AuthTokenAccessException(
                    "Could not find an implementation of the "
                        + ScpCfAuthTokenFacade.class.getName()
                        + " to determine the Xsuaa Service Token."));
    }

    @SuppressWarnings( "deprecation" )
    @Nonnull
    private Try<ScpCfTenant> tryGetTenantFromServiceBinding(
        @Nonnull final Try<com.sap.cloud.sdk.cloudplatform.CloudPlatform> platform )
    {
        for( final ServiceBindingTenantExtractor extractor : ServiceBindingTenantExtractor.values() ) {
            final Try<Iterable<JsonObject>> bindings =
                platform
                    .map(com.sap.cloud.sdk.cloudplatform.ScpCfCloudPlatform.class::cast)
                    .map(p -> p.getServiceCredentialsByPlan(extractor.getService(), filter -> true).values())
                    .map(Iterables::concat);

            if( bindings.isFailure() ) {
                log.debug("Unable to parse service bindings for {}.", extractor.getService(), bindings.getCause());
                continue;
            }

            final Optional<ScpCfTenant> tenant =
                Streams
                    .stream(bindings.get())
                    .peek(obj -> log.trace("Trying to extract tenant information from service binding {}.", obj))
                    .flatMap(obj -> extractor.getExtractor().apply(obj).toJavaStream())
                    .findFirst();

            if( tenant.isPresent() ) {
                return Try.success(tenant.get());
            }
        }
        return Try.failure(new CloudPlatformException("Failed to extract tenant from service bindings."));
    }
}
