/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.datamodel.odata.client;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.UUID;
import java.util.function.Function;

import javax.annotation.Nonnull;

import com.google.common.annotations.Beta;

import lombok.Getter;

/**
 * The {@code ODataProtocol} defines all necessary information that is needed in order to differentiate between
 * different OData protocol versions.
 */
@Beta
public interface ODataProtocol extends ODataResponseDescriptor, ODataLiteralSerializer
{
    /**
     * Version 2.0 of the OData protocol.
     */
    ODataProtocol V2 = new ODataProtocolV2();

    /**
     * Version 4.0 of the OData protocol.
     */
    ODataProtocol V4 = new ODataProtocolV4();

    /**
     * The version number of this protocol.
     * 
     * @return A string representing the OData version, e.g. "4.0"
     */
    @Nonnull
    String getProtocolVersion();

    /**
     * OData protocol v2
     */
    @Beta
    final class ODataProtocolV2 implements ODataProtocol
    {
        @Getter
        private final String protocolVersion = "2.0";
        @Getter
        private final String[] pathToResultSet = { "d", "results" };
        @Getter
        private final String[] pathToResultSingle = { "d" };
        // The * is the function import name, but not passed through since it does not really matter.
        @Getter
        private final String[] pathToResultPrimitive = { "d", "*" };
        @Getter
        private final String[] pathToInlineCount = { "__count" };

        @Getter
        private final Function<Number, String> NumberSerializer = ODataProtocolV2::numberToString;
        @Getter
        private final Function<UUID, String> UUIDSerializer = v -> String.format("guid'%s'", v);
        @Getter
        private final Function<OffsetDateTime, String> DateTimeOffsetSerializer =
            v -> String.format("datetimeoffset'%s'", v.format(DateTimeFormatter.ISO_OFFSET_DATE_TIME));
        @Getter
        private final Function<LocalTime, String> TimeOfDaySerializer =
            v -> String.format("time'%s'", Duration.ofNanos(v.toNanoOfDay()));
        @Getter
        private final Function<LocalDateTime, String> DateTimeSerializer =
            v -> String.format("datetime'%s'", v.format(DateTimeFormatter.ISO_LOCAL_DATE_TIME));

        private static String numberToString( final Number n )
        {
            final String result;

            if( n instanceof Integer ) {
                result = Integer.toString(n.intValue());
            } else if( n instanceof Short ) {
                result = Short.toString(n.shortValue());
            } else if( n instanceof Byte ) {
                result = Byte.toString(n.byteValue());
            } else if( n instanceof Long ) {
                result = n.longValue() + "L";
            } else if( n instanceof Float ) {
                result = n.floatValue() + "f";
            } else if( n instanceof Double ) {
                result = n.doubleValue() + "d";
            } else if( n instanceof BigDecimal ) {
                result = ((BigDecimal) n).toPlainString() + "M";
            } else {
                final String message =
                    String.format(
                        "Unrecognized number type: %s. Should be one of: Integer, Long, Float, Double, BigDecimal.",
                        n.getClass());
                throw new IllegalStateException(message);
            }
            return result;
        }
    }

    /**
     * OData protocol v4
     */
    @Beta
    final class ODataProtocolV4 implements ODataProtocol
    {
        @Getter
        private final String protocolVersion = "4.0";
        @Getter
        private final String[] pathToResultSet = { "value" };
        @Getter
        private final String[] pathToResultSingle = {};
        @Getter
        private final String[] pathToResultPrimitive = { "value" };
        @Getter
        private final String[] pathToInlineCount = { "@odata.count" };

        @Getter
        private final Function<Number, String> NumberSerializer = Number::toString;
        @Getter
        private final Function<UUID, String> UUIDSerializer = UUID::toString;
        @Getter
        private final Function<OffsetDateTime, String> DateTimeOffsetSerializer =
            v -> v.format(DateTimeFormatter.ISO_OFFSET_DATE_TIME);
        @Getter
        private final Function<LocalTime, String> TimeOfDaySerializer = v -> v.format(DateTimeFormatter.ISO_LOCAL_TIME);
        @Getter
        private final Function<LocalDateTime, String> DateTimeSerializer =
            v -> DateTimeOffsetSerializer.apply(v.atOffset(ZoneOffset.UTC));
    }
}
