/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.datamodel.odata.client;

import java.io.IOException;

import javax.annotation.Nonnull;

import com.google.common.annotations.Beta;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonPrimitive;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonToken;

import io.vavr.control.Option;

/**
 * Implementation to deserialize OData responses based on a given {@link ODataProtocol}.
 */
@Beta
public class ODataResponseDeserializer
{
    private final ODataProtocol protocol;

    /**
     * Create a deserializer for a given {@link ODataProtocol}.
     * 
     * @param protocol
     *            The {@link ODataProtocol} to adhere to.
     */
    public ODataResponseDeserializer( @Nonnull final ODataProtocol protocol )
    {
        this.protocol = protocol;
    }

    /**
     * Position the {@see JsonReader} to the response result set.
     * 
     * @param reader
     *            The internal JsonReader instance.
     * @throws IOException
     *             If response cannot be read.
     */
    public void positionReaderToResultSet( @Nonnull final JsonReader reader )
        throws IOException
    {
        final String[] path = protocol.getPathToResultSet();
        reader.beginObject();
        for( int i = 0; i < path.length; i++ ) {
            while( reader.peek() == JsonToken.NAME && !path[i].equals(reader.nextName()) ) {
                JsonParser.parseReader(reader);
            }
            if( i < path.length - 1 ) {
                reader.beginObject();
            }
        }
        reader.beginArray();
    }

    /**
     * Get the element to the response result set.
     * 
     * @param element
     *            The root element.
     * @return The optional result as JsonArray.
     */
    @Nonnull
    public Option<JsonArray> getElementToResultSet( @Nonnull final JsonElement element )
    {
        final JsonElement resultElement = getResultJsonElement(element, protocol.getPathToResultSet());
        return Option.of(resultElement).filter(JsonElement::isJsonArray).map(JsonElement::getAsJsonArray);
    }

    /**
     * Get the element to the single response result item.
     * 
     * @param element
     *            The root element.
     * @return The optional result as JsonObject.
     */
    @Nonnull
    public Option<JsonObject> getElementToResultSingle( @Nonnull final JsonElement element )
    {
        final JsonElement resultElement = getResultJsonElement(element, protocol.getPathToResultSingle());
        return Option.of(resultElement).filter(JsonElement::isJsonObject).map(JsonElement::getAsJsonObject);
    }

    /**
     * Get the element to the response result set.
     * 
     * @param element
     *            The root element.
     * @return The optional result as JsonArray.
     */
    @Nonnull
    public Option<JsonArray> getElementToResultPrimitiveSet( @Nonnull final JsonElement element )
    {
        final JsonElement resultElement = getResultJsonElement(element, protocol.getPathToResultPrimitive());
        return Option.of(resultElement).filter(JsonElement::isJsonArray).map(JsonElement::getAsJsonArray);
    }

    /**
     * Get the element to the single response result item.
     * 
     * @param element
     *            The root element.
     * @return The optional result as JsonPrimitive.
     */
    @Nonnull
    public Option<JsonPrimitive> getElementToResultPrimitiveSingle( @Nonnull final JsonElement element )
    {
        final JsonElement resultElement = getResultJsonElement(element, protocol.getPathToResultPrimitive());
        return Option.of(resultElement).filter(JsonElement::isJsonPrimitive).map(JsonElement::getAsJsonPrimitive);
    }

    private JsonElement getResultJsonElement( @Nonnull final JsonElement element, final String[] path )
    {
        JsonElement resultElement = element;
        for( int i = 0; i < path.length && element != null; i++ ) {
            if( path[i].equals("*") ) {
                if( !resultElement.getAsJsonObject().entrySet().isEmpty() ) {
                    resultElement = resultElement.getAsJsonObject().entrySet().iterator().next().getValue();
                } else {
                    return null;
                }
            } else {
                resultElement = resultElement.getAsJsonObject().get(path[i]);
            }
        }
        return resultElement;
    }
}
