/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.datamodel.odata.client.request;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import com.google.common.annotations.Beta;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;
import com.sap.cloud.sdk.datamodel.odata.client.expression.Expressions;

/**
 * Wrapper class to hold an OData entity composite key.
 */
@Beta
public class ODataEntityKey
{
    @Nonnull
    private final Map<String, Expressions.OperandSingle> compositeKey = new LinkedHashMap<>();

    /**
     * Get the field names that are part of the composite key.
     * 
     * @return A set of entity property field names.
     */
    @Nonnull
    public Set<String> getFieldNames()
    {
        return compositeKey.keySet();
    }

    /**
     * Add an entity property to this composite key.
     *
     * @param propertyName
     *            Name of the property (derived from the EDMX)
     * @param value
     *            Property value, assumed to be a primitive.
     * @param <PrimitiveT>
     *            Type of the primitive value.
     * @throws IllegalArgumentException
     *             When there is no mapping found for the provided Java literal.
     */
    public <PrimitiveT> void addKeyProperty( @Nonnull final String propertyName, @Nonnull final PrimitiveT value )
    {
        compositeKey.put(propertyName, Expressions.createOperand(value));
    }

    /**
     * Add properties to the OData entity key.
     * 
     * @param properties
     *            The key-value mapping.
     * @return The same instance.
     */
    @Nonnull
    public ODataEntityKey addKeyProperties( @Nonnull final Map<String, Object> properties )
    {
        properties.forEach(this::addKeyProperty);
        return this;
    }

    /**
     * Create an instance of {@link ODataEntityKey} from a generic key-value composition.
     * 
     * @param key
     *            Key-value pairs for primitive entity properties (derived from the EDMX)
     * @throws IllegalArgumentException
     *             When there is no mapping found for the provided Java literal.
     * @return A new instance of {@link ODataEntityKey}.
     */
    @Nonnull
    public static ODataEntityKey of( @Nonnull final Map<String, Object> key )
    {
        final ODataEntityKey result = new ODataEntityKey();
        key.forEach(( k, v ) -> {
            if( v == null ) {
                throw new IllegalArgumentException("Key value is invalid: property \"" + k + "\" must not be null.");
            }
            result.addKeyProperty(k, v);
        });
        return result;
    }

    /**
     * Serializes key properties into an OData URL format for entity keys.
     *
     * @param protocol
     *            The OData protocol to derive serialization rules from.
     * @return Encoded URL string representation of entity key. These characters are encoded if present:'%','
     *         ','/','$','&','#' and '?'.
     */
    @Nonnull
    public String toString( @Nonnull final ODataProtocol protocol )
    {
        final StringBuilder urlString = new StringBuilder();

        if( !compositeKey.isEmpty() ) {
            urlString.append('(');

            if( compositeKey.size() == 1 ) {
                final Expressions.OperandSingle singleValue = compositeKey.values().iterator().next();
                urlString.append(ODataUriFactory.encodeParameter(singleValue.getExpression(protocol)));
            } else {
                urlString.append(
                    compositeKey
                        .entrySet()
                        .stream()
                        .map(
                            keyProperty -> keyProperty.getKey()
                                + '='
                                + ODataUriFactory.encodeParameter(keyProperty.getValue().getExpression(protocol)))
                        .collect(Collectors.joining(",")));
            }

            urlString.append(')');
        }

        return urlString.toString();
    }
}
