/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.datamodel.odata.client.request;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import com.google.common.annotations.Beta;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;
import com.sap.cloud.sdk.datamodel.odata.client.expression.Expressions;

/**
 * Wrapper class to hold function parameters.
 */
@Beta
public class ODataFunctionParameters
{
    @Nonnull
    private final Map<String, Expressions.OperandSingle> typeAwareParameters = new LinkedHashMap<>();

    /**
     * Add a parameter to function parameters
     *
     * @param parameterName
     *            Name of the property (derived from the EDMX)
     * @param value
     *            Property value, assumed to be a primitive.
     * @param <PrimitiveT>
     *            Type of the primitive value.
     */
    public <PrimitiveT> void addParameter( @Nonnull final String parameterName, @Nonnull final PrimitiveT value )
    {
        typeAwareParameters.put(parameterName, Expressions.createOperand(value));
    }

    /**
     * Serializes all parameters into an OData URL format.
     *
     * @param protocol
     *            The OData protocol to derive serialization rules from.
     * @return Encoded URL string representation of function parameters. These characters are encoded if present:'%','
     *         ','/','$','&','#' and '?'.
     */
    @Nonnull
    public String toString( @Nonnull final ODataProtocol protocol )
    {
        final StringBuilder urlString = new StringBuilder();

        if( !typeAwareParameters.isEmpty() ) {
            urlString.append('(');

            urlString.append(

                typeAwareParameters
                    .entrySet()
                    .stream()
                    .map(
                        parameter -> parameter.getKey()
                            + '='
                            + ODataUriFactory.encodeParameter(parameter.getValue().getExpression(protocol)))
                    .collect(Collectors.joining(",")));

            urlString.append(')');
        }

        return urlString.toString();
    }
}
