/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.datamodel.odata.client.request;

import java.net.URI;

import javax.annotation.Nonnull;

import org.apache.http.client.HttpClient;

import com.google.common.annotations.Beta;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;

import lombok.EqualsAndHashCode;
import lombok.Getter;

/**
 * The executable OData action request.
 */
@Getter
@EqualsAndHashCode( callSuper = true )
@Beta
public class ODataRequestAction extends ODataRequestGeneric
{
    @Nonnull
    private final String actionName;

    @Nonnull
    private final String actionParameters;

    {
        addHeader("Content-Type", "application/json");
    }

    /**
     * Default constructor for OData Action request.
     *
     * @param servicePath
     *            The OData service path.
     * @param actionName
     *            The OData action name.
     * @param actionParameters
     *            The action parameters HTTP payload.
     * @param protocol
     *            The OData protocol to use.
     */
    public ODataRequestAction(
        @Nonnull final String servicePath,
        @Nonnull final String actionName,
        @Nonnull final String actionParameters,
        @Nonnull final ODataProtocol protocol )
    {
        super(protocol, servicePath);
        this.actionName = actionName;
        this.actionParameters = actionParameters;
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public URI getRelativeUri()
    {
        return ODataUriFactory.getUriWithEncodedQuery(getServicePath(), getActionName(), getRequestQuery());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Nonnull
    public ODataRequestResultGeneric execute( @Nonnull final HttpClient httpClient )
    {
        final ODataHttpRequest request = new ODataHttpRequest(this, httpClient, actionParameters);
        return tryExecuteWithCsrfToken(httpClient, request::requestPost).get();
    }
}
