/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.datamodel.odata.client.request;

import java.net.URI;

import javax.annotation.Nonnull;

import org.apache.http.client.HttpClient;

import com.google.common.annotations.Beta;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;

import lombok.EqualsAndHashCode;
import lombok.Getter;

/**
 * The executable OData create request.
 */
@Getter
@EqualsAndHashCode( callSuper = true )
@Beta
public class ODataRequestCreate extends ODataRequestGeneric
{
    @Nonnull
    private final String entityName;

    @Nonnull
    private final String serializedEntity;

    {
        addHeader("Content-Type", "application/json");
        addHeader("Accept", "application/json");
    }

    /**
     * Default constructor for OData Create request.
     *
     * @param servicePath
     *            The OData service path.
     * @param entityName
     *            The OData entity name.
     * @param serializedEntity
     *            The serialized query payload.
     * @param protocol
     *            The OData protocol to use.
     */
    public ODataRequestCreate(
        @Nonnull final String servicePath,
        @Nonnull final String entityName,
        @Nonnull final String serializedEntity,
        @Nonnull final ODataProtocol protocol )
    {
        super(protocol, servicePath);
        this.entityName = entityName;
        this.serializedEntity = serializedEntity;
    }

    @Nonnull
    @Override
    public URI getRelativeUri()
    {
        return ODataUriFactory.getUriWithEncodedQuery(getServicePath(), getEntityName(), getRequestQuery());
    }

    @Nonnull
    @Override
    public ODataRequestResultGeneric execute( @Nonnull final HttpClient httpClient )
    {
        final ODataHttpRequest request = new ODataHttpRequest(this, httpClient, serializedEntity);

        return tryExecuteWithCsrfToken(httpClient, request::requestPost).get();
    }
}
