/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.datamodel.odata.client.request;

import java.net.URI;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.HttpClient;

import com.google.common.annotations.Beta;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

/**
 * The executable OData delete request.
 */
@Getter
@EqualsAndHashCode( callSuper = true )
@Beta
@Slf4j
public class ODataRequestDelete extends ODataRequestGeneric
{
    @Nonnull
    private final String entityName;

    @Nonnull
    private final String entityKey;

    @Nullable
    private final String versionIdentifier;

    {
        addHeader("Accept", "application/json");
    }

    /**
     * Default constructor for OData Delete request.
     *
     * @param servicePath
     *            The OData service path.
     * @param entityName
     *            The entity name.
     * @param entityKey
     *            The entity key.
     * @param versionIdentifier
     *            The version identifier.
     * @param protocol
     *            The OData protocol to use.
     */
    public ODataRequestDelete(
        @Nonnull final String servicePath,
        @Nonnull final String entityName,
        @Nonnull final String entityKey,
        @Nullable final String versionIdentifier,
        @Nonnull final ODataProtocol protocol )
    {
        super(protocol, servicePath);
        this.entityName = entityName;
        this.entityKey = entityKey;
        this.versionIdentifier = versionIdentifier;
    }

    @Nonnull
    @Override
    public URI getRelativeUri()
    {
        return ODataUriFactory
            .getUriWithEncodedQueryAndParameters(getServicePath(), getEntityName(), getRequestQuery(), getEntityKey());
    }

    @Nonnull
    @Override
    public ODataRequestResultGeneric execute( @Nonnull final HttpClient httpClient )
    {
        final ODataHttpRequest request = new ODataHttpRequest(this, httpClient);

        addVersionIdentifierToHeaderIfPresent(versionIdentifier);

        return tryExecuteWithCsrfToken(httpClient, request::requestDelete).get();
    }
}
