/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.datamodel.odata.client.request;

import java.net.URI;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.HttpClient;

import com.google.common.annotations.Beta;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;

import lombok.EqualsAndHashCode;
import lombok.Getter;

/**
 * The executable OData function request.
 */
@Getter
@EqualsAndHashCode( callSuper = true )
@Beta
public class ODataRequestFunction extends ODataRequestGeneric
{
    @Nonnull
    private final String functionName;

    @Nullable
    private final String parameters;

    /**
     * Default constructor for OData Function request.
     *
     * @param servicePath
     *            The OData service path.
     * @param functionName
     *            The OData function name.
     * @param parameters
     *            The function parameters as HTTP query.
     * @param protocol
     *            The OData protocol to use.
     */
    public ODataRequestFunction(
        @Nonnull final String servicePath,
        @Nonnull final String functionName,
        @Nullable final String parameters,
        @Nonnull final ODataProtocol protocol )
    {
        super(protocol, servicePath);
        this.functionName = functionName;
        this.parameters = parameters;
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public URI getRelativeUri()
    {
        return ODataUriFactory.getUriWithEncodedQueryAndParameters(
            getServicePath(),
            getFunctionName(),
            getRequestQuery(),
            getParameters());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Nonnull
    public ODataRequestResultGeneric execute( @Nonnull final HttpClient httpClient )
    {
        final ODataHttpRequest request = new ODataHttpRequest(this, httpClient);
        return tryExecute(request::requestGet).get();
    }
}
