/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.datamodel.odata.client.request;

import java.net.URI;

import javax.annotation.Nonnull;

import org.apache.http.client.HttpClient;

import com.google.common.annotations.Beta;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;

import lombok.EqualsAndHashCode;
import lombok.Getter;

/**
 * The result type of the OData read by key request.
 */
@Getter
@EqualsAndHashCode( callSuper = true )
@Beta
public class ODataRequestReadByKey extends ODataRequestGeneric
{
    @Nonnull
    private final String entityName;

    @Nonnull
    private final String entityKey;

    @Nonnull
    private final String queryString;

    /**
     * Default constructor for OData Read request.
     *
     * @param servicePath
     *            The OData service path.
     * @param entityName
     *            The OData entity name.
     * @param entityKey
     *            The entity key.
     * @param queryString
     *            The HTTP query.
     * @param protocol
     *            The OData protocol to use.
     *
     */
    public ODataRequestReadByKey(
        @Nonnull final String servicePath,
        @Nonnull final String entityName,
        @Nonnull final String entityKey,
        @Nonnull final String queryString,
        @Nonnull final ODataProtocol protocol )
    {
        super(protocol, servicePath);
        this.entityName = entityName;
        this.entityKey = entityKey;
        this.queryString = queryString;
    }

    @Nonnull
    @Override
    public URI getRelativeUri()
    {
        return ODataUriFactory
            .getUriWithEncodedQueryAndParameters(getServicePath(), getEntityName(), getRequestQuery(), getEntityKey());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Nonnull
    public String getRequestQuery()
    {
        final String genericQueryString = super.getRequestQuery();
        if( !genericQueryString.isEmpty() && !queryString.isEmpty() ) {
            return queryString + "&" + genericQueryString;
        }
        return queryString + genericQueryString;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Nonnull
    public ODataRequestResultGeneric execute( @Nonnull final HttpClient httpClient )
    {
        final ODataHttpRequest request = new ODataHttpRequest(this, httpClient);
        return tryExecute(request::requestGet).get();
    }
}
