/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.datamodel.odata.client.request;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.commons.lang3.ClassUtils;
import org.apache.http.Header;
import org.apache.http.HeaderElement;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.entity.BufferedHttpEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.util.EntityUtils;

import com.google.common.annotations.Beta;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.reflect.TypeToken;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonToken;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;
import com.sap.cloud.sdk.datamodel.odata.client.ODataResponseDeserializer;
import com.sap.cloud.sdk.datamodel.odata.client.exception.ODataDeserializationException;
import com.sap.cloud.sdk.datamodel.odata.client.exception.ODataException;
import com.sap.cloud.sdk.datamodel.odata.client.exception.ODataResponseException;
import com.sap.cloud.sdk.datamodel.odata.client.exception.ODataServiceError;
import com.sap.cloud.sdk.datamodel.odata.client.exception.ODataServiceErrorException;
import com.sap.cloud.sdk.result.GsonResultElementFactory;
import com.sap.cloud.sdk.result.ResultCollection;
import com.sap.cloud.sdk.result.ResultElement;
import com.sap.cloud.sdk.result.ResultObject;
import com.sap.cloud.sdk.result.ResultPrimitive;

import io.vavr.CheckedFunction1;
import io.vavr.control.Option;
import io.vavr.control.Try;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

/**
 * OData request result for reading entities.
 */
@Slf4j
@Beta
@EqualsAndHashCode
public class ODataRequestResultGeneric
    implements
    ODataRequestResult,
    ODataRequestResultDeserializable,
    Iterable<ResultElement>
{
    private static final Charset DEFAULT_CHARSET = StandardCharsets.UTF_8;

    private final ODataProtocol protocol;
    private final ODataResponseDeserializer deserializer;

    private volatile BufferedHttpEntity bufferedHttpEntity;
    private volatile boolean isBufferHttpEntity = true;

    @Getter
    @Nonnull
    private final ODataRequestGeneric oDataRequest;

    @Getter
    @Nonnull
    private final HttpResponse httpResponse;

    public ODataRequestResultGeneric(
        @Nonnull final ODataRequestGeneric oDataRequest,
        @Nonnull final HttpResponse httpResponse )
    {
        this.oDataRequest = oDataRequest;
        this.httpResponse = httpResponse;

        protocol = oDataRequest.getProtocol();
        deserializer = new ODataResponseDeserializer(protocol);
    }

    /**
     * Get the original map of HTTP response header key-values.
     *
     * @return The response headers.
     */
    @Nonnull
    protected Map<String, Iterable<String>> getResponseHeaders()
    {
        return Arrays.stream(getHttpResponse().getAllHeaders()).collect(
            Collectors.toMap(
                header -> header.getName().toLowerCase(),
                header -> Arrays.stream(header.getElements()).map(HeaderElement::getName).collect(Collectors.toList()),
                Iterables::concat));
    }

    /**
     * Method that allows consumers to disable buffering HTTP response entity. Note that once this is disabled, HTTP
     * responses can only be streamed/read once
     *
     */
    public void disableBufferingHttpResponse()
    {
        if( bufferedHttpEntity == null ) {
            isBufferHttpEntity = false;
        } else {
            log.warn("Buffering the HTTP response cannot be disabled! The content has already been buffered.");
        }
    }

    /**
     * Protocol dependent method to consume the InputStream from the HTTP response entity.
     *
     * @param streamConsumer
     *            The consumer of the InputStream.
     * @param <T>
     *            The generic return type.
     * @return The response.
     * @throws ODataDeserializationException
     *             When streamed deserialization process failed.
     */
    @Nullable
    private <T> T consumeHttpEntityContent( @Nonnull final CheckedFunction1<InputStream, T> streamConsumer )
    {
        final HttpEntity entity = getHttpEntity();
        if( entity == null ) {
            final String msg = "OData response does not contain an HTTP entity.";
            log.warn(msg);
            throw new ODataDeserializationException(getODataRequest(), getHttpResponse(), msg, null);
        }

        try( InputStream content = entity.getContent() ) {
            return streamConsumer.apply(content);
        }
        catch( final IOException e ) {
            final String msg = "OData response entity stream cannot established.";
            log.debug(msg, e);
            throw new ODataDeserializationException(getODataRequest(), getHttpResponse(), msg, e);
        }
        catch( final UnsupportedOperationException e ) {
            final String msg = "OData response entity content cannot be represented as stream object.";
            log.debug(msg, e);
            throw new ODataDeserializationException(getODataRequest(), getHttpResponse(), msg, e);
        }
        // CHECKSTYLE:OFF
        catch( final Throwable e ) {
            final String msg = "A problem occurred while streaming the OData response.";
            log.debug(msg, e);
            throw new ODataDeserializationException(getODataRequest(), getHttpResponse(), msg, e);
        }
        // CHECKSTYLE:ON
    }

    /**
     * Method that creates a {@link BufferedHttpEntity} from the {@link HttpEntity} if buffering the HTTP response is
     * not turned off by using {@link ODataRequestResultGeneric#disableBufferingHttpResponse()}
     *
     * @return An HttpEntity
     */
    private HttpEntity getHttpEntity()
    {

        if( bufferedHttpEntity != null ) {
            return bufferedHttpEntity;
        }
        if( isBufferHttpEntity && bufferedHttpEntity == null ) {
            synchronized( this ) {
                if( isBufferHttpEntity && bufferedHttpEntity == null ) {
                    try {
                        bufferedHttpEntity = new BufferedHttpEntity(getHttpResponse().getEntity());
                    }
                    catch( final IOException e ) {
                        final String msg = "OData response entity cannot be buffered";
                        log.debug(msg, e);
                        throw new ODataDeserializationException(getODataRequest(), getHttpResponse(), msg, e);
                    }
                }
            }
            return bufferedHttpEntity;
        }
        return getHttpResponse().getEntity();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void streamElements( @Nonnull final Consumer<ResultElement> handler )
    {
        final GsonResultElementFactory resultElementFactory =
            new GsonResultElementFactory(ODataGsonBuilder.newGsonBuilder());
        final Integer numConsumedElements = consumeHttpEntityContent(inputStream -> {
            final JsonReader reader = new JsonReader(new InputStreamReader(inputStream, DEFAULT_CHARSET));
            deserializer.positionReaderToResultSet(reader);

            int count = 0;
            while( reader.hasNext() && reader.peek() == JsonToken.BEGIN_OBJECT ) {
                final JsonElement jsonElement = JsonParser.parseReader(reader);
                final ResultElement resultElement = resultElementFactory.create(jsonElement);
                handler.accept(resultElement);
                count++;
            }
            reader.close();
            return count;
        });

        log.debug("Iterated {} elements.", numConsumedElements);
    }

    /**
     * Try to extract a version identifier from the ETag header.
     *
     * @return An option holding the version identifier or {@link Option.None}, if none was found.
     */
    @Nonnull
    public Option<String> getVersionIdentifierFromHeader()
    {
        return Option.ofOptional(
            Stream
                .of(getHttpResponse().getHeaders("ETag"))
                .filter(h -> "ETag".equalsIgnoreCase(h.getName()))
                .map(Header::getValue)
                .filter(s -> !Strings.isNullOrEmpty(s))
                .findFirst());
    }

    @Nonnull
    private ResultPrimitive loadPrimitiveFromResponse()
    {
        final GsonResultElementFactory elementFactory = new GsonResultElementFactory(ODataGsonBuilder.newGsonBuilder());
        final ResultPrimitive result = consumeHttpEntityContent(inputStream -> {
            final JsonElement element = JsonParser.parseReader(new InputStreamReader(inputStream, DEFAULT_CHARSET));
            final Option<ResultPrimitive> single =
                deserializer.getElementToResultPrimitiveSingle(element).map(elementFactory::create).map(
                    ResultElement::getAsPrimitive);
            return single.getOrNull();
        });
        if( result == null ) {
            log.debug("OData response cannot be read as a primitive value.");
            throw new ODataDeserializationException(
                getODataRequest(),
                getHttpResponse(),
                "Unable to read OData response.",
                null);
        }
        return result;
    }

    @Nonnull
    private ResultCollection loadPrimitiveCollectionFromResponse()
    {
        final GsonResultElementFactory elementFactory = new GsonResultElementFactory(ODataGsonBuilder.newGsonBuilder());
        final ResultCollection result = consumeHttpEntityContent(inputStream -> {
            final JsonElement element = JsonParser.parseReader(new InputStreamReader(inputStream, DEFAULT_CHARSET));
            final Option<ResultCollection> set =
                deserializer.getElementToResultPrimitiveSet(element).map(elementFactory::create).map(
                    ResultElement::getAsCollection);
            return set.getOrNull();
        });
        if( result == null ) {
            log.debug("OData response cannot be read as set of primitive values.");
            throw new ODataDeserializationException(
                getODataRequest(),
                getHttpResponse(),
                "Unable to read OData response.",
                null);
        }
        return result;
    }

    @Nonnull
    private ResultObject loadEntryFromResponse()
    {
        final GsonResultElementFactory elementFactory = new GsonResultElementFactory(ODataGsonBuilder.newGsonBuilder());
        final ResultObject result = consumeHttpEntityContent(inputStream -> {
            final JsonElement element = JsonParser.parseReader(new InputStreamReader(inputStream, DEFAULT_CHARSET));
            final Option<ResultObject> single =
                deserializer.getElementToResultSingle(element).map(elementFactory::create).map(
                    ResultElement::getAsObject);
            return single.getOrNull();
        });
        if( result == null ) {
            log.debug("OData response cannot be read as a single entity.");
            throw new ODataDeserializationException(
                getODataRequest(),
                getHttpResponse(),
                "Unable to read OData response.",
                null);
        }
        return result;
    }

    @Nullable
    private ResultObject loadErrorFromResponse()
    {
        final GsonResultElementFactory elementFactory = new GsonResultElementFactory(ODataGsonBuilder.newGsonBuilder());
        return consumeHttpEntityContent(inputStream -> {
            final JsonElement element = JsonParser.parseReader(new InputStreamReader(inputStream, DEFAULT_CHARSET));
            if( element.isJsonNull() ) {
                log.debug("OData response did not contain an error.");
                return null;
            }
            return elementFactory.create(element).getAsObject();
        });
    }

    @Nonnull
    private ResultCollection loadEntryCollectionFromResponse()
    {
        final GsonResultElementFactory elementFactory = new GsonResultElementFactory(ODataGsonBuilder.newGsonBuilder());
        final ResultCollection result = consumeHttpEntityContent(inputStream -> {
            final JsonElement element = JsonParser.parseReader(new InputStreamReader(inputStream, DEFAULT_CHARSET));
            final Option<ResultCollection> set =
                deserializer.getElementToResultSet(element).map(elementFactory::create).map(
                    ResultElement::getAsCollection);
            return set.getOrNull();
        });
        if( result == null ) {
            log.debug("OData response cannot be read as set of entities.");
            throw new ODataDeserializationException(
                getODataRequest(),
                getHttpResponse(),
                "Unable to read OData response.",
                null);
        }
        return result;
    }

    @Nonnull
    private ResultElement getResultElement()
    {
        final ResultElement result = loadEntryFromResponse();
        checkForODataError(result);
        return result;
    }

    @Nonnull
    Iterable<ResultElement> getResultElements()
    {
        final ResultCollection result = loadEntryCollectionFromResponse();
        checkForODataError(result);
        return result;
    }

    @Nonnull
    @Override
    public Iterator<ResultElement> iterator()
    {
        return getResultElements().iterator();
    }

    @Override
    @Nonnull
    public <T> T as( @Nonnull final Class<T> objectType )
    {
        if( isPrimitiveOrWrapperOrString(objectType) ) {
            final ContentType contentType = ContentType.get(getHttpResponse().getEntity());

            // parse text/plain responses directly, do not use JSON deserializers
            if( Objects.equals(contentType.getMimeType(), ContentType.TEXT_PLAIN.getMimeType()) ) {
                return getPrimitiveObjectFromPlainText(objectType);
            }

            return getPrimitiveObjectFromJson(objectType);
        } else {
            return getComplexObjectFromJson(objectType);
        }
    }

    @Nonnull
    private <T> T getComplexObjectFromJson( @Nonnull final Class<T> objectType )
    {
        final ResultObject resultObject = loadEntryFromResponse();
        checkForODataError(resultObject);

        final ODataRequestGeneric r = getODataRequest();

        return Try
            .of(() -> resultObject.as(objectType))
            .onFailure(e -> log.debug("Failed to deserialize {} from JSON response.", objectType))
            .getOrElseThrow(
                e -> new ODataDeserializationException(
                    r,
                    getHttpResponse(),
                    "Failed to deserialize a complex object.",
                    e));
    }

    @Nonnull
    private <T> T getPrimitiveObjectFromJson( @Nonnull final Class<T> objectType )
    {
        final ResultPrimitive resultPrimitive = loadPrimitiveFromResponse();
        checkForODataError(resultPrimitive);

        final ODataRequestGeneric r = getODataRequest();

        return Try
            .of(() -> getPrimitiveAsType(resultPrimitive, objectType))
            .onFailure(e -> log.debug("Failed to deserialize {} from JSON response.", objectType))
            .getOrElseThrow(
                e -> new ODataDeserializationException(
                    r,
                    getHttpResponse(),
                    "Failed to deserialize a primitive object.",
                    e));
    }

    @Nonnull
    private <T> T getPrimitiveObjectFromPlainText( @Nonnull final Class<T> objectType )
        throws ODataDeserializationException
    {
        final ODataRequestGeneric r = getODataRequest();
        final HttpResponse httpResponse = getHttpResponse();

        final String objectText =
            Try.of(() -> EntityUtils.toString(getHttpEntity())).getOrElseThrow(
                e -> new ODataDeserializationException(r, httpResponse, "Failed to parse HTTP response.", e));

        return Try
            .of(() -> new Gson().fromJson(objectText, objectType))
            .filterTry(
                Objects::nonNull,
                () -> new ODataDeserializationException(r, httpResponse, "The response is null.", null))
            .onFailure(e -> log.debug("Failed to deserialize {} from text/plain response: {}", objectType, objectText))
            .getOrElseThrow(
                e -> new ODataDeserializationException(
                    r,
                    httpResponse,
                    "Failed to deserialize a primitive object.",
                    e));
    }

    /*
     * Helper function to check if the objectType passed in any of the primitive or wrapper types (Boolean, Byte,
     * Character, Short, Integer, Long, Double, Float) or is String.
     */
    private boolean isPrimitiveOrWrapperOrString( @Nonnull final Class<?> objectType )
    {
        return ClassUtils.isPrimitiveOrWrapper(objectType) || objectType == String.class;
    }

    @Nonnull
    @SuppressWarnings( "unchecked" )
    private <T> T as( @Nonnull final Type objectType )
    {
        final Class<T> typeClass;
        if( objectType instanceof ParameterizedType ) {
            typeClass = (Class<T>) ((ParameterizedType) objectType).getRawType();
        } else {
            typeClass = (Class<T>) objectType.getClass();
        }
        return as(typeClass);
    }

    @Override
    @Nonnull
    public <T> List<T> asList( @Nonnull final Class<T> objectType )
    {
        final ResultCollection result =
            isPrimitiveOrWrapperOrString(objectType)
                ? loadPrimitiveCollectionFromResponse()
                : loadEntryCollectionFromResponse();

        checkForODataError(result);

        return Try
            .of(() -> result.asList(objectType))
            .onFailure(e -> log.debug("Failed to parse OData result to a list of {}", objectType))
            .getOrElseThrow(
                e -> new ODataDeserializationException(
                    getODataRequest(),
                    getHttpResponse(),
                    "Failed to parse OData result to a list.",
                    e));
    }

    @Nonnull
    @SuppressWarnings( "unchecked" )
    private <T> List<T> asList( @Nonnull final Type objectType )
    {
        final Class<T> typeClass;
        if( objectType instanceof ParameterizedType ) {
            typeClass = (Class<T>) ((ParameterizedType) objectType).getRawType();
        } else {
            typeClass = (Class<T>) objectType.getClass();
        }
        return asList(typeClass);
    }

    @Override
    public long getInlineCount()
    {
        final String[] path = protocol.getPathToInlineCount();
        ResultElement resultElement = getResultElement().getAsObject().get(path[0]);
        for( int i = 1; i < path.length && resultElement != null && resultElement.isResultObject(); i++ ) {
            resultElement = resultElement.getAsObject().get(path[i]);
        }

        if( resultElement == null ) {
            final String message = "Inline count not found in OData response payload.";
            throw new ODataDeserializationException(oDataRequest, getHttpResponse(), message, null);
        }
        return resultElement.getAsPrimitive().asLong();
    }

    @Override
    @Nonnull
    public Map<String, Object> asMap()
    {
        final Type type = new TypeToken<Map<String, Object>>()
        {
            private static final long serialVersionUID = 42L;
        }.getType();
        return as(type);
    }

    @Override
    @Nonnull
    public List<Map<String, Object>> asListOfMaps()
    {
        final Type type = new TypeToken<Map<String, Object>>()
        {
            private static final long serialVersionUID = 42L;
        }.getType();
        return asList(type);
    }

    @Override
    @Nonnull
    public Iterable<String> getHeaderValues( @Nonnull final String headerName )
    {
        return getResponseHeaders().getOrDefault(headerName.toLowerCase(), Collections.emptyList());
    }

    @Override
    @Nonnull
    public Iterable<String> getHeaderNames()
    {
        return getResponseHeaders().keySet();
    }

    @Nonnull
    private <T> T getPrimitiveAsType( @Nonnull final ResultPrimitive primitive, @Nonnull final Class<T> type )
        throws IllegalArgumentException
    {
        final Object primitiveAsType;
        try {
            if( type == Boolean.class ) {
                primitiveAsType = primitive.asBoolean();
            } else if( type == Byte.class ) {
                primitiveAsType = primitive.asByte();
            } else if( type == Short.class ) {
                primitiveAsType = primitive.asShort();
            } else if( type == Integer.class ) {
                primitiveAsType = primitive.asInteger();
            } else if( type == Long.class ) {
                primitiveAsType = primitive.asLong();
            } else if( type == BigInteger.class ) {
                primitiveAsType = primitive.asBigInteger();
            } else if( type == Float.class ) {
                primitiveAsType = primitive.asFloat();
            } else if( type == Double.class ) {
                primitiveAsType = primitive.asDouble();
            } else if( type == BigDecimal.class ) {
                primitiveAsType = primitive.asBigDecimal();
            } else if( type == Character.class ) {
                primitiveAsType = primitive.asCharacter();
            } else if( type == String.class ) {
                primitiveAsType = primitive.asString();
            } else {
                throw new IllegalArgumentException(
                    "Failed to convert primitive '"
                        + primitive.asString()
                        + "' to unsupported type: "
                        + type.getName()
                        + ".");
            }
        }
        catch( final UnsupportedOperationException e ) {
            throw new IllegalArgumentException(
                "Failed to convert primitive '" + primitive.asString() + "' to type: " + type.getName() + ".",
                e);
        }

        @SuppressWarnings( "unchecked" )
        final T result = (T) primitiveAsType;
        return result;
    }

    /**
     * Check the HTTP response code and body of the OData request result. If the code indicates an unhealthy response,
     * an exception will be thrown with further details.
     *
     * @throws ODataResponseException
     *             When the response code infers an unhealthy state, i.e. when >= 400.
     * @throws ODataServiceErrorException
     *             When the response contains an OData error message according to specification.
     */
    protected void requireHealthyResponse()
    {
        final HttpResponse httpResponse = getHttpResponse();
        final StatusLine statusLine = httpResponse.getStatusLine();

        if( statusLine != null && statusLine.getStatusCode() < 400 ) {
            return;
        }
        final ODataResponseException preparedException =
            new ODataResponseException(
                getODataRequest(),
                httpResponse,
                "The HTTP response code indicates an error.",
                null);
        try {
            final ResultElement resultElement = loadErrorFromResponse();
            if( resultElement != null ) {
                checkForODataError(resultElement);
            } else {
                throw preparedException;
            }
        }
        catch( final ODataServiceErrorException serviceError ) {
            throw serviceError;
        }
        // We cannot attach an ODataDeserializationException as suppressed exception to another ODataException
        // that is called self-suppression and not permitted
        catch( final ODataException otherOdataException ) {
            log.warn(
                "An error occurred when attempting to deserialize an OData error from the response.",
                otherOdataException);
            throw preparedException;
        }
        catch( final Exception other ) {
            preparedException.addSuppressed(other);
            throw preparedException;
        }
    }

    /**
     * Check the received OData response object for an error message.
     *
     * @param result
     *            The OData response object.
     * @throws ODataServiceErrorException
     *             When the received OData service response contained a parsable error message.
     * @throws ODataDeserializationException
     *             When the received OData service response contained a non-parsable error message.
     */
    private void checkForODataError( @Nonnull final ResultElement result )
    {
        if( !result.isResultObject() ) {
            return;
        }
        final ResultElement errorElement = result.getAsObject().get("error");
        if( errorElement == null || !errorElement.isResultObject() ) {
            return;
        }
        log.debug("Found error field in OData response.");
        final ResultObject errorObject = errorElement.getAsObject();
        try {
            final ODataServiceError oDataServiceError =
                ODataServiceError.fromResultObject(errorObject, getODataRequest().getProtocol());
            final String errorMessage = "The OData service responded with an error message.";
            throw new ODataServiceErrorException(
                getODataRequest(),
                getHttpResponse(),
                errorMessage,
                null,
                oDataServiceError);
        }
        catch( final UnsupportedOperationException e ) {
            log.debug("Failed to deserialize error property from OData response: {}", errorElement, e);
            throw new ODataDeserializationException(
                getODataRequest(),
                getHttpResponse(),
                "Unable to deserialize OData error.",
                e);
        }
    }
}
