/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.datamodel.odata.client.request;

import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.util.EntityUtils;

import com.google.common.annotations.Beta;
import com.google.common.collect.Iterables;
import com.sap.cloud.sdk.datamodel.odata.client.exception.ODataDeserializationException;
import com.sap.cloud.sdk.datamodel.odata.client.exception.ODataResponseException;

import io.vavr.Tuple2;
import io.vavr.control.Try;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

/**
 * OData request result for reading entities.
 */
@Slf4j
@Beta
public class ODataRequestResultMultipartGeneric implements ODataRequestResultMultipart, ODataRequestResult
{
    private static final String HEADER_VALUE_DELIMITER = ";";

    private static final String MSG_ON_FAILURE =
        "Failed to read response entity from OData request result. Please make sure the entity is not consumed already by the application.";

    @Nonnull
    private final ODataRequestBatch batchRequest;

    @Getter
    @Nonnull
    private final HttpResponse httpResponse;

    @Getter( lazy = true, value = AccessLevel.PRIVATE )
    @Nullable
    private final String batchContent = readAndConsumeResponseContent();

    /**
     * Create an instance of OData request result for multipart/mixed responses.
     * 
     * @param oDataRequest
     *            The original OData request instance.
     * @param httpResponse
     *            The native HTTP response object.
     */
    ODataRequestResultMultipartGeneric(
        @Nonnull final ODataRequestBatch oDataRequest,
        @Nonnull final HttpResponse httpResponse )
    {
        this.batchRequest = oDataRequest;
        this.httpResponse = httpResponse;
    }

    /**
     * Get the original map of HTTP response header key-values.
     *
     * @return The response headers.
     */
    @Nonnull
    private Map<String, Iterable<String>> getResponseHeaders()
    {
        return Arrays.stream(getHttpResponse().getAllHeaders()).collect(
            Collectors.toMap(
                header -> header.getName().toLowerCase(),
                header -> Arrays.asList(header.getValue().split("\\W*" + HEADER_VALUE_DELIMITER + "\\W*")),
                Iterables::concat));
    }

    @Override
    @Nonnull
    public Iterable<String> getHeaderValues( @Nonnull final String headerName )
    {
        return getResponseHeaders().getOrDefault(headerName.toLowerCase(), Collections.emptyList());
    }

    @Override
    @Nonnull
    public Iterable<String> getHeaderNames()
    {
        return getResponseHeaders().keySet();
    }

    /**
     * Get the original {@link ODataRequestBatch batch request} that was used for running the OData request.
     *
     * @return The batch request this
     */
    @Nonnull
    @Override
    public ODataRequestExecutable getODataRequest()
    {
        return batchRequest;
    }

    /**
     * @throws ODataResponseException
     *             When the OData batch response cannot be parsed.
     * @throws IllegalArgumentException
     *             When the provided request reference could not be found in the original batch request.
     */
    @Nonnull
    @Override
    public ODataRequestResultGeneric getResult( @Nonnull final ODataRequestGeneric request )
        throws ODataResponseException,
            IllegalArgumentException
    {
        @Nullable
        final Tuple2<Integer, Integer> responsePosition = ODataRequestBatch.getBatchItemPosition(batchRequest, request);
        if( responsePosition == null ) {
            throw new IllegalArgumentException(
                "Incorrect API usage. Please pass the original OData request reference that was handled as batch request item.");
        }

        final String content = getBatchContent();
        if( content == null ) {
            final String msg = "OData batch response did not contain an HTTP body.";
            throw new ODataResponseException(batchRequest, httpResponse, msg, null);
        }

        final Iterable<String> contentType = getHeaderValues("Content-Type");
        final ODataRequestResultMultipartParser parser = new ODataRequestResultMultipartParser(content, contentType);

        log.debug("Looking for request {} in batch response at position {}", request, responsePosition);
        final HttpResponse response =
            responsePosition._2() == null
                ? parser.extractHttpResponseForReceiving(responsePosition._1())
                : parser.extractHttpResponseForModifying(responsePosition._1(), responsePosition._2());

        if( response == null ) {
            final String msg = "Illegal payload for OData batch response item.";
            throw new ODataDeserializationException(batchRequest, httpResponse, msg, null);
        }
        return new ODataRequestResultGeneric(request, response);
    }

    /**
     * Check the HTTP response code of the OData request result. If the code indicates an unhealthy response, an
     * exception will be thrown with further details.
     *
     * @throws ODataResponseException
     *             When the response code infers an unhealthy state, i.e. when >= 400
     */
    void requireHealthyResponse()
    {
        final HttpResponse httpResponse = getHttpResponse();
        final StatusLine statusLine = httpResponse.getStatusLine();
        if( statusLine != null && statusLine.getStatusCode() >= 400 ) {
            final String msg = "HTTP response code of OData service batch request contains an error.";
            throw new ODataResponseException(batchRequest, httpResponse, msg, null);
        }
    }

    @Nullable
    private String readAndConsumeResponseContent()
    {
        return Try
            .of(() -> EntityUtils.toString(httpResponse.getEntity()))
            .onFailure(e -> log.error(MSG_ON_FAILURE, e))
            .getOrNull();
    }
}
