/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cloud.sdk.datamodel.odata.client.request;

import java.net.URI;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.HttpClient;

import com.google.common.annotations.Beta;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * The executable OData patch update request.
 */
@Getter
@EqualsAndHashCode( callSuper = true )
@Beta
@Slf4j
public class ODataRequestUpdate extends ODataRequestGeneric
{
    @Nonnull
    private final String entityName;

    @Nonnull
    private final String entityKey;

    @Nonnull
    private final String serializedEntity;

    {
        addHeader("Accept", "application/json");
    }

    /**
     * The {@link UpdateStrategy} determines if the entity will be changed or replaced.
     */
    @Nonnull
    @Setter
    private UpdateStrategy updateStrategy;

    @Nullable
    private final String versionIdentifier;

    {
        addHeader("Content-Type", "application/json");
    }

    /**
     * Default constructor for OData Update request.
     *
     * @param servicePath
     *            The OData service path.
     * @param entityName
     *            The OData entity name.
     * @param entityKey
     *            The OData entity key.
     * @param serializedEntity
     *            The serialized OData entity.
     * @param updateStrategy
     *            The update strategy.
     * @param versionIdentifier
     *            The entity version identifier.
     * @param protocol
     *            The OData protocol to use.
     */
    public ODataRequestUpdate(
        @Nonnull final String servicePath,
        @Nonnull final String entityName,
        @Nonnull final String entityKey,
        @Nonnull final String serializedEntity,
        @Nonnull final UpdateStrategy updateStrategy,
        @Nullable final String versionIdentifier,
        @Nonnull final ODataProtocol protocol )
    {
        super(protocol, servicePath);
        this.entityName = entityName;
        this.entityKey = entityKey;
        this.serializedEntity = serializedEntity;
        this.updateStrategy = updateStrategy;
        this.versionIdentifier = versionIdentifier;
    }

    @Nonnull
    @Override
    public URI getRelativeUri()
    {
        return ODataUriFactory
            .getUriWithEncodedQueryAndParameters(getServicePath(), getEntityName(), getRequestQuery(), getEntityKey());
    }

    @Nonnull
    @Override
    public ODataRequestResultGeneric execute( @Nonnull final HttpClient httpClient )
    {
        final ODataHttpRequest request = new ODataHttpRequest(this, httpClient, serializedEntity);
        addVersionIdentifierToHeaderIfPresent(versionIdentifier);

        if( updateStrategy == UpdateStrategy.MODIFY_WITH_PATCH ) {
            return tryExecuteWithCsrfToken(httpClient, request::requestPatch).get();
        } else if( updateStrategy == UpdateStrategy.REPLACE_WITH_PUT ) {
            return tryExecuteWithCsrfToken(httpClient, request::requestPut).get();
        } else {
            throw new IllegalStateException("Unexpected update Strategy: " + updateStrategy);
        }
    }
}
