/*
 * Copyright (c) 2020 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cloud.sdk.datamodel.odata.client.request;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.function.Predicate;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.annotations.Beta;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;

import io.vavr.control.Option;
import lombok.extern.slf4j.Slf4j;

/**
 * Builds up OData URLs and ensures correct encoding.
 */
@Beta
@Slf4j
public class ODataUriFactory
{
    private static final Predicate<String> VALID_URL_QUERY =
        Pattern.compile("^[a-zA-Z0-9/?:@\\-._~!$&'()*+,;=%]*$").asPredicate();

    private static final Map<String, String> ENCODING_MAP_FOR_SERVICE_AND_ENTITY_PATH =
        ImmutableMap
            .<String, String> builder()
            .put("%", "%25")
            .put("#", "%23")
            .put(" ", "%20")
            .put("\\?", "%3F")
            .put("\\\\", "%5C")
            .put("\"", "%22")
            .build();

    private static final Map<String, String> ENCODING_MAP_FOR_QUERY =
        ImmutableMap
            .<String, String> builder()
            .put("%", "%25")
            .put("\\$", "%24")
            .put(" ", "%20")
            .put("&", "%26")
            .put("#", "%23")
            .put("\\?", "%3F")
            .put("\\\\", "%5C")
            .put("\"", "%22")
            .build();

    private static final Map<String, String> ENCODING_MAP_FOR_PARAMETERS =
        ImmutableMap
            .<String, String> builder()
            .put("%", "%25")
            .put("/", "%2F")
            .put("\\$", "%24")
            .put(" ", "%20")
            .put("&", "%26")
            .put("#", "%23")
            .put("\\?", "%3F")
            .put("\\\\", "%5C")
            .put("\"", "%22")
            .build();

    /**
     * Constructs an URI out of service path, entity path and query string.
     *
     * @return The URI
     * @throws IllegalArgumentException
     *             When the URI could not constructed.
     */
    static URI getUriWithEncodedQuery(
        @Nonnull final String servicePath,
        @Nonnull final String entityPath,
        @Nullable final String encodedQuery )
    {
        return getUriWithEncodedQueryAndParameters(servicePath, entityPath, encodedQuery, null);
    }

    /**
     * Constructs an URI out of service path, entity path, parameters and query string.
     *
     * @return The URI
     * @throws IllegalArgumentException
     *             When the URI could not constructed.
     */
    static URI getUriWithEncodedQueryAndParameters(
        @Nonnull final String servicePath,
        @Nonnull final String entityPath,
        @Nullable final String encodedQuery,
        @Nullable final String encodedParameters )
    {
        final Option<String> maybeQueryEncoded = Option.of(encodedQuery).filter(s -> !Strings.isNullOrEmpty(s));
        if( maybeQueryEncoded.isDefined() && !maybeQueryEncoded.exists(VALID_URL_QUERY) ) {
            throw new IllegalArgumentException(
                "The query part of OData request is not correctly encoded: \"" + encodedQuery + "\"");
        }

        final Option<String> maybeParametersEncoded =
            Option.of(encodedParameters).filter(s -> !Strings.isNullOrEmpty(s));

        final String servicePathProcessed = encodeServicePath(servicePath);
        final String entityPathProcessed = encodePath(entityPath);

        String resultUrl = servicePathProcessed + entityPathProcessed;

        if( maybeParametersEncoded.isDefined() ) {
            resultUrl += maybeParametersEncoded.get();
        }

        resultUrl += maybeQueryEncoded.map(q -> "?" + q).getOrElse("");

        try {
            return new URI(resultUrl);
        }
        catch( final URISyntaxException e ) {
            log.error(
                "Failed to construct URI for OData request with request path '{}', entity path '{}' and query '{}'.",
                servicePath,
                entityPath,
                encodedQuery,
                e);
            throw new IllegalArgumentException("Failed to construct URI.", e);
        }
    }

    @Nonnull
    private static String encodeServicePath( @Nonnull final String servicePath )
    {
        final String pathWithSlashes = "/" + servicePath + "/";
        return encodePath(pathWithSlashes.replaceAll("//+", "/"));
    }

    /**
     * Encodes the parameter passed with encoding set listed in {@link ODataUriFactory#ENCODING_MAP_FOR_PARAMETERS} Used
     * in {@link com.sap.cloud.sdk.datamodel.odata.client.request.ODataEntityKey} and
     * {@link com.sap.cloud.sdk.datamodel.odata.client.request.ODataFunctionParameters} for encoding parameters
     *
     * @param input
     *            The key or function parameter of the request
     * @return The encoded parameter
     */
    @Nonnull
    static String encodeParameter( @Nonnull final String input )
    {
        return encode(input, ENCODING_MAP_FOR_PARAMETERS);
    }

    /**
     * Encodes the query passed with encoding set listed in {@link ODataUriFactory#ENCODING_MAP_FOR_QUERY}
     *
     * @param input
     *            The query string of the request
     * @return The encoded query
     */
    @Nonnull
    public static String encodeQuery( @Nonnull final String input )
    {
        return encode(input, ENCODING_MAP_FOR_QUERY);
    }

    /**
     * Encodes the service and entity path in a URI created from
     * {@link ODataUriFactory#getUriWithEncodedQueryAndParameters} with encoding set listed in
     * {@link ODataUriFactory#ENCODING_MAP_FOR_SERVICE_AND_ENTITY_PATH}
     *
     * @param input
     *            The service or entity path of the request
     * @return The encoded service or entity path
     */
    @Nonnull
    private static String encodePath( @Nonnull final String input )
    {
        return encode(input, ENCODING_MAP_FOR_SERVICE_AND_ENTITY_PATH);
    }

    private static String encode( @Nonnull String input, final Map<String, String> encodingMap )
    {
        for( final Map.Entry<String, String> entry : encodingMap.entrySet() ) {
            input = input.replaceAll(entry.getKey(), entry.getValue());
        }
        return input;
    }
}
