/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.datamodel.odata.client.expression;

import javax.annotation.Nonnull;

import com.google.common.annotations.Beta;

/**
 * Collection operations for generic OData filter expression operands.
 */
@Beta
public interface FilterableCollection extends Expressions.Operand
{
    /**
     * Filter by expression "hasSubset".
     *
     * @param operand
     *            Only operand of collection type.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueBoolean.Expression hasSubset( @Nonnull final ValueCollection operand )
    {
        return FilterExpressionCollection.hasSubset(this::getExpression, operand::getExpression);
    }

    /**
     * Filter by expression "hasSubset".
     *
     * @param operand
     *            Only operand of Java iterable.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueBoolean.Expression hasSubset( @Nonnull final Iterable<?> operand )
    {
        final ValueCollection value = ValueCollection.literal(operand);
        return FilterExpressionCollection.hasSubset(this::getExpression, value);
    }

    /**
     * Filter by expression "hasSubSequence".
     *
     * @param operand
     *            Only operand of collection type.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueBoolean.Expression hasSubSequence( @Nonnull final ValueCollection operand )
    {
        final Expressions.OperandMultiple value = operand::getExpression;
        return FilterExpressionCollection.hasSubSequence(this::getExpression, value);
    }

    /**
     * Filter by expression "hasSubSequence".
     *
     * @param operand
     *            Only operand of Java iterable.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueBoolean.Expression hasSubSequence( @Nonnull final Iterable<?> operand )
    {
        final ValueCollection value = ValueCollection.literal(operand);
        return FilterExpressionCollection.hasSubSequence(this::getExpression, value);
    }

    /**
     * Filter by expression "contains".
     *
     * @param operand
     *            Only operand of collection type.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueBoolean.Expression contains( @Nonnull final ValueCollection operand )
    {
        final Expressions.OperandMultiple value = operand::getExpression;
        return FilterExpressionCollection.contains(this::getExpression, value);
    }

    /**
     * Filter by expression "contains".
     *
     * @param operand
     *            Only operand of Java iterable.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueBoolean.Expression contains( @Nonnull final Iterable<?> operand )
    {
        final Expressions.OperandMultiple value = ValueCollection.literal(operand);
        return FilterExpressionCollection.contains(this::getExpression, value);
    }

    /**
     * Filter by expression "startsWith".
     *
     * @param operand
     *            Only operand of collection type.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueBoolean.Expression startsWith( @Nonnull final ValueCollection operand )
    {
        final Expressions.OperandMultiple value = operand::getExpression;
        return FilterExpressionCollection.startsWith(this::getExpression, value);
    }

    /**
     * Filter by expression "startsWith".
     *
     * @param operand
     *            Only operand of Java iterable.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueBoolean.Expression startsWith( @Nonnull final Iterable<?> operand )
    {
        final ValueCollection value = ValueCollection.literal(operand);
        return FilterExpressionCollection.startsWith(this::getExpression, value);
    }

    /**
     * Filter by expression "endsWith".
     *
     * @param operand
     *            Only operand of collection type.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueBoolean.Expression endsWith( @Nonnull final ValueCollection operand )
    {
        final Expressions.OperandMultiple value = operand::getExpression;
        return FilterExpressionCollection.endsWith(this::getExpression, value);
    }

    /**
     * Filter by expression "endsWith".
     *
     * @param operand
     *            Only operand of Java iterable.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueBoolean.Expression endsWith( @Nonnull final Iterable<?> operand )
    {
        final ValueCollection value = ValueCollection.literal(operand);
        return FilterExpressionCollection.endsWith(this::getExpression, value);
    }

    /**
     * Filter by expression "indexOf".
     *
     * @param operand
     *            Only operand of collection type.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueNumeric.Expression indexOf( @Nonnull final ValueCollection operand )
    {
        final Expressions.OperandMultiple value = operand::getExpression;
        return FilterExpressionCollection.indexOf(this::getExpression, value);
    }

    /**
     * Filter by expression "indexOf".
     *
     * @param operand
     *            Only operand of Java iterable.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueNumeric.Expression indexOf( @Nonnull final Iterable<?> operand )
    {
        final ValueCollection value = ValueCollection.literal(operand);
        return FilterExpressionCollection.indexOf(this::getExpression, value);
    }

    /**
     * Filter by expression "concat".
     *
     * @param operand
     *            Only operand of collection type.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueCollection.Expression concat( @Nonnull final ValueCollection operand )
    {
        final Expressions.OperandMultiple value = operand::getExpression;
        return FilterExpressionCollection.concat(this::getExpression, value);
    }

    /**
     * Filter by expression "concat".
     *
     * @param operand
     *            Only operand of Java iterable.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueCollection.Expression concat( @Nonnull final Iterable<?> operand )
    {
        final ValueCollection value = ValueCollection.literal(operand);
        return FilterExpressionCollection.concat(this::getExpression, value);
    }

    /**
     * Filter by expression "length".
     *
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueNumeric.Expression length()
    {
        return FilterExpressionCollection.length(this::getExpression);
    }

    /**
     * Filter by expression "substring".
     *
     * @param operand
     *            Only operand of Integer type.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueCollection.Expression substring( @Nonnull final Integer operand )
    {
        final ValueNumeric value = ValueNumeric.literal(operand);
        return FilterExpressionCollection.substring(this::getExpression, value);
    }

    /**
     * Filter by expression "substring".
     *
     * @param operandIndex
     *            Operand of Integer type to mark the start of the subset.
     * @param operandLength
     *            Operand of Integer type to mark the size of the subset.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default
        ValueCollection.Expression
        substring( @Nonnull final Integer operandIndex, @Nonnull final Integer operandLength )
    {
        final ValueNumeric value1 = ValueNumeric.literal(operandIndex);
        final ValueNumeric value2 = ValueNumeric.literal(operandLength);
        return FilterExpressionCollection.substring(this::getExpression, value1, value2);
    }

    /**
     * Filter by lambda expression "all".
     *
     * @param operand
     *            Operand to provide a generic filter to the collection item.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueBoolean.Expression all( @Nonnull final ValueBoolean operand )
    {
        final String lambdaFieldPrefix = "a";
        final Expressions.OperandMultiple thisMulti = this::getExpression;
        return FilterExpressionCollection.all(thisMulti, operand::getExpression, o -> true, lambdaFieldPrefix);
    }

    /**
     * Filter by lambda expression "any".
     *
     * @param operand
     *            Operand to provide a generic filter to the collection item.
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueBoolean.Expression any( @Nonnull final ValueBoolean operand )
    {
        final String lambdaFieldPrefix = "a";
        final Expressions.OperandMultiple thisMulti = this::getExpression;
        return FilterExpressionCollection.any(thisMulti, operand::getExpression, o -> true, lambdaFieldPrefix);
    }

    /**
     * Filter by lambda expression "any", for finding non-empty collections.
     *
     * @return The FluentHelper filter.
     */
    @Nonnull
    default ValueBoolean.Expression any()
    {
        final Expressions.OperandMultiple thisMulti = this::getExpression;
        return FilterExpressionCollection.any(thisMulti);
    }
}
