/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.datamodel.odata.client.request;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.HttpVersion;
import org.apache.http.StatusLine;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicHttpResponse;
import org.apache.http.message.BasicStatusLine;

import com.google.common.collect.Iterables;
import com.google.common.collect.Streams;

import io.vavr.control.Option;
import io.vavr.control.Try;
import lombok.RequiredArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;

@RequiredArgsConstructor
@Slf4j
class ODataRequestResultMultipartParser
{
    private static final String BATCH_NEW_LINE = "\r\n";
    private static final Pattern PATTERN_BOUNDARY_DELIMITER = Pattern.compile("boundary=([\\w_-]+)");

    @Nonnull
    final String batchContent;

    @Nonnull
    final Iterable<String> contentType;

    /**
     * Extract the HttpResponse from the OData batch result for an item at the given position.
     * 
     * @param positionInBatchMultipart
     *            The position within the batch response.
     * @return A newly created instance of {@code HttpResponse} or {@code null} when there was an issue.
     */
    @Nullable
    HttpResponse extractHttpResponseForReceiving( final int positionInBatchMultipart )
    {
        @Nullable
        final String batchDelimiter = getDelimiterFromHeader();

        @Nullable
        final String selectedResponse = getPayloadFromBoundary(batchContent, positionInBatchMultipart, batchDelimiter);

        @Nullable
        final BatchItemPayload payload = createBatchItemPayload(selectedResponse);

        return createHttpResponse(payload);
    }

    @Nullable
    HttpResponse extractHttpResponseForModifying( final int positionInBatchMultipart, final int positionInChangeset )
    {
        @Nullable
        final String batchDelimiter = getDelimiterFromHeader();

        @Nullable
        final String changeset = getPayloadFromBoundary(batchContent, positionInBatchMultipart, batchDelimiter);

        @Nullable
        final String changesetDelimiter = getDelimiterFromChangeset(changeset);

        // fallback if changeset cannot be found at the expected position because the there is an error
        if( changesetDelimiter == null ) {
            return extractHttpResponseForReceiving(positionInBatchMultipart);
        }

        @Nullable
        String selectedResponse = getPayloadFromBoundary(changeset, positionInChangeset, changesetDelimiter);

        // fallback if changeset item could not be retrieved because the whole changeset has an error
        if( selectedResponse == null && positionInChangeset > 0 && hasChangesetFailed(changeset, changesetDelimiter) ) {
            selectedResponse = getPayloadFromBoundary(changeset, 0, changesetDelimiter);
        }

        @Nullable
        final BatchItemPayload payload = createBatchItemPayload(selectedResponse);

        return createHttpResponse(payload);
    }

    // from here on helper methods

    private boolean hasChangesetFailed( @Nullable final String content, @Nullable final String changesetDelimiter )
    {
        if( content == null || changesetDelimiter == null ) {
            return false;
        }
        final boolean changesetHasOnlyOneEntry = StringUtils.countMatches(content, changesetDelimiter) == 2;
        if( changesetHasOnlyOneEntry ) {
            final String payloadFromFirstItem = getPayloadFromBoundary(content, 0, changesetDelimiter);
            if( payloadFromFirstItem != null ) {
                final BatchItemPayload firstItem = createBatchItemPayload(payloadFromFirstItem);
                if( firstItem != null ) {
                    final Try<BasicStatusLine> firstStatusLine = constructStatusLineFromBatchItem(firstItem);
                    if( !firstStatusLine.isFailure() ) {
                        final int statusCode = firstStatusLine.get().getStatusCode();
                        if( statusCode >= HttpStatus.SC_BAD_REQUEST ) {
                            log.debug(
                                "The requested batch item in changeset of OData batch response could not be found due to a previous error response. That error response will be returned instead.");
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    }

    @Nullable
    private String getDelimiterFromChangeset( @Nullable final String changeset )
    {
        if( changeset == null ) {
            return null;
        }
        final Matcher matcher = PATTERN_BOUNDARY_DELIMITER.matcher(changeset);
        if( !matcher.find() ) {
            log.debug(
                "Payload batch item delimiter for changeset \"{}\" could not be extracted from OData batch response except.",
                changeset);
            return null;
        }
        return "--" + matcher.group(1);
    }

    @Nullable
    private String getDelimiterFromHeader()
    {
        if( !Iterables.contains(contentType, "multipart/mixed") ) {
            log.warn("Illegal value in HTTP header \"Content-Type\" of OData batch response.");
        }

        final Optional<String> boundary =
            Streams.stream(contentType).map(PATTERN_BOUNDARY_DELIMITER::matcher).filter(Matcher::find).findFirst().map(
                m -> m.group(1));

        if( !boundary.isPresent() ) {
            log.error("A boundary delimiter cannot be resolved from OData batch response header.");
            return null;
        }
        return "--" + boundary.get();
    }

    @Nonnull
    private Try<BasicStatusLine> constructStatusLineFromBatchItem( @Nonnull final BatchItemPayload payload )
    {
        return Try.of(() -> {
            final String[] status = payload.getResponseStatus().split(" ", 3);
            final Optional<HttpVersion> version =
                Stream
                    .of(HttpVersion.HTTP_0_9, HttpVersion.HTTP_1_0, HttpVersion.HTTP_1_1)
                    .filter(v -> status[0].equalsIgnoreCase(v.toString()))
                    .findFirst();
            if( !version.isPresent() ) {
                log.error("Unexpected HTTP version found in OData batch response item: {}", status[0]);
                throw new IllegalStateException("Unexpected HTTP version.");
            }
            final int code = Integer.parseInt(status[1]);
            final String reason = status[2];
            return new BasicStatusLine(version.get(), code, reason);
        });
    }

    @Nullable
    private BatchItemPayload createBatchItemPayload( @Nullable final String responseItem )
    {
        if( responseItem == null ) {
            log.debug("Skipping the payload deserialization because of previous issues.");
            return null;
        }
        final String[] splitResponse = responseItem.split(BATCH_NEW_LINE + BATCH_NEW_LINE, 3);
        if( splitResponse.length < 2 ) {
            log.error("OData batch response item is malformed.");
            return null;
        }

        final List<String> batchItemHeaders = Arrays.asList(splitResponse[0].split(BATCH_NEW_LINE));

        final String[] responseHead = splitResponse[1].split(BATCH_NEW_LINE, 2);
        final String responseStatus = responseHead[0];
        final List<String> responseHeaders =
            responseHead.length > 1 ? Arrays.asList(responseHead[1].split(BATCH_NEW_LINE)) : Collections.emptyList();

        final Option<String> content =
            splitResponse.length > 2 ? Option.of(splitResponse[2]).filter(StringUtils::isNotBlank) : Option.none();
        return new BatchItemPayload(batchItemHeaders, responseStatus, responseHeaders, content);
    }

    @Nullable
    private static
        String
        getPayloadFromBoundary( @Nullable final String content, final int position, @Nullable final String delimiter )
    {
        if( delimiter == null || content == null ) {
            log.debug("Skipping the payload extraction because of previous issues.");
            return null;
        }

        for( int i = 0, p = content.indexOf(delimiter); i < position + 1; i++ ) {
            final int newPosition = content.indexOf(delimiter, p + 1);
            if( i == position ) {
                if( p < 0 || newPosition < 0 ) {
                    break;
                }
                return content.substring(p + delimiter.length(), newPosition).trim();
            }
            p = newPosition;
        }

        log.error(
            "Payload batch item with delimiter \"{}\" could not be extracted from OData batch response at position {}.",
            delimiter,
            position);
        return null;
    }

    @Nullable
    private HttpResponse createHttpResponse( @Nullable final BatchItemPayload payload )
    {
        if( payload == null ) {
            log.debug("Skipping the HttpResponse generation because of previous issues.");
            return null;
        }

        // status
        final Try<BasicStatusLine> maybeStatusLine = constructStatusLineFromBatchItem(payload);
        final StatusLine statusLine = maybeStatusLine.onFailure(e -> {
            log.error(
                "Failed to construct status line for OData batch response item: {}",
                payload.getResponseStatus(),
                e);
        }).getOrElse(() -> new BasicStatusLine(HttpVersion.HTTP_1_1, 200, "Ok"));

        // headers
        final BasicHttpResponse httpResponse = new BasicHttpResponse(statusLine);
        for( final String header : payload.getBatchItemHeaders() ) {
            final String[] headerParts = header.split(":", 2);
            httpResponse.setHeader(headerParts[0], headerParts[1]);
        }

        // content
        payload.getContent().peek(
            content -> httpResponse.setEntity(new StringEntity(content, ContentType.APPLICATION_JSON)));

        return httpResponse;
    }

    @Value
    private static class BatchItemPayload
    {
        @Nonnull
        List<String> batchItemHeaders;
        @Nonnull
        String responseStatus;
        @Nonnull
        List<String> responseHeaders;
        @Nonnull
        Option<String> content;
    }
}
