/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cloud.sdk.datamodel.odata.client.request;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.annotations.Beta;
import com.google.common.base.Strings;
import com.google.common.escape.Escaper;
import com.google.common.net.PercentEscaper;
import com.google.common.net.UrlEscapers;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;
import com.sap.cloud.sdk.datamodel.odata.client.expression.ODataResourcePath;

import io.vavr.control.Option;
import lombok.extern.slf4j.Slf4j;

/**
 * Builds up OData URLs and ensures correct encoding.
 */
@Beta
@Slf4j
public class ODataUriFactory
{
    private static final Predicate<String> VALID_URL_QUERY =
        Pattern.compile("^[a-zA-Z0-9/?:@\\-._~!$&'()*+,;=%]*$").asPredicate();

    /**
     * The characters which are not encoded in the query part of a URL during URL encoding
     */
    public static final char[] SAFE_CHARS_IN_QUERY = { '_', '*', '+', '-', ':', ',', '/', '\'', '(', ')', '.', '|' };

    @SuppressWarnings( "UnstableApiUsage" )
    private static final Escaper URL_QUERY_ESCAPER = new PercentEscaper(new String(SAFE_CHARS_IN_QUERY), false);

    /**
     * Constructs a URI out of service path, entity path and query string.
     * 
     * @param servicePath
     *            The unencoded service path.
     * @param resourcePath
     *            The {@link ODataResourcePath path} identifying the resource to be accessed.
     * @param encodedQuery
     *            Optional: An encoded string representing the URL query part.
     * @param protocol
     *            The {@link ODataProtocol} the URL should conform to.
     *
     * @return The correctly encoded URI.
     */
    @Nonnull
    static URI createAndEncodeUri(
        @Nonnull final String servicePath,
        @Nonnull final ODataResourcePath resourcePath,
        @Nullable final String encodedQuery,
        @Nonnull final ODataProtocol protocol )
    {
        final String encodedResourcePath = resourcePath.toEncodedPathString();
        return createAndEncodeUri(servicePath, encodedResourcePath, encodedQuery);
    }

    /**
     * Constructs a URI out of service path, entity path and query string.
     *
     * @param servicePath
     *            The unencoded service path.
     * @param encodedResourcePath
     *            The encoded resource path identifying the resource to be accessed.
     * @param encodedQuery
     *            Optional: An encoded string representing the URL query part.
     *
     * @return The correctly encoded URI.
     */
    @Nonnull
    static URI createAndEncodeUri(
        @Nonnull final String servicePath,
        @Nonnull final String encodedResourcePath,
        @Nullable final String encodedQuery )
    {
        String encodedPath = encodePath(servicePath);
        encodedPath = sanitizeUrlPath(encodedPath);

        encodedPath += encodedResourcePath.startsWith("/") ? encodedResourcePath : "/" + encodedResourcePath;

        final Option<String> maybeQueryEncoded = Option.of(encodedQuery).filter(s -> !Strings.isNullOrEmpty(s));

        if( maybeQueryEncoded.isDefined() && !maybeQueryEncoded.exists(VALID_URL_QUERY) ) {
            throw new IllegalArgumentException(
                "The query part of OData request is not correctly encoded: \"" + encodedQuery + "\"");
        }

        final String resultUrl = encodedPath + maybeQueryEncoded.map(q -> "?" + q).getOrElse("");

        try {
            return new URI(resultUrl);
        }
        catch( final URISyntaxException e ) {
            log.error(
                "Failed to construct URI for OData request with service path '{}', resource path '{}' and query '{}'.",
                servicePath,
                encodedResourcePath,
                encodedQuery,
                e);
            throw new IllegalArgumentException("Failed to construct URI.", e);
        }
    }

    /**
     * Encodes the individual path parts of a string. Forward slashes are treated as path segment separators and thus
     * not be encoded. Encoding is done by {@link UrlEscapers#urlPathSegmentEscaper()}.
     *
     * @param path
     *            The unencoded URL path.
     *
     * @return The percentage encoded URL path.
     */
    @Nonnull
    public static String encodePath( @Nonnull final String path )
    {
        return Arrays.stream(path.split("/")).map(ODataUriFactory::encodePathSegment).collect(Collectors.joining("/"));
    }

    /**
     * Encodes an individual part of a URL path. Any forward slashes will not be treated as path segment separators but
     * instead be encoded. Encoding is done by {@link UrlEscapers#urlPathSegmentEscaper()}.
     *
     * @param path
     *            The unencoded URL path segment.
     *
     * @return The percentage encoded URL path segment.
     */
    @Nonnull
    public static String encodePathSegment( @Nonnull final String path )
    {
        return UrlEscapers.urlPathSegmentEscaper().escape(path);
    }

    /**
     * Encodes all characters in the query except ones listed in {@link ODataUriFactory#SAFE_CHARS_IN_QUERY}
     *
     * @param input
     * 
     *            The query string of the request
     * @return The encoded query
     */
    @Nonnull
    public static String encodeQuery( @Nonnull final String input )
    {
        return URL_QUERY_ESCAPER.escape(input);
    }

    /**
     * Brings any string into the form "/A/B/C". The path will contain no double slashes, always start with a slash and
     * never end with a slash. An empty path will stay empty.
     * 
     * @param path
     *            The path to be sanitized.
     *
     * @return The sanitized path according to the rules above.
     */
    @Nonnull
    private static String sanitizeUrlPath( @Nonnull final String path )
    {
        final String pathWithPrefixingSlash = "/" + path;
        final String pathWithoutDoubleSlashes = pathWithPrefixingSlash.replaceAll("//+", "/");
        return pathWithoutDoubleSlashes.replaceAll("/$", "");
    }
}
