/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.datamodel.odata.client.request;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.annotations.Beta;
import com.sap.cloud.sdk.datamodel.odata.client.ODataProtocol;
import com.sap.cloud.sdk.datamodel.odata.client.expression.Expressions;

import io.vavr.Tuple;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.RequiredArgsConstructor;

/**
 * Abstract class to build parameter expressions for the URL path. Parameters can resemble an entity key or function
 * parameters.
 */
@Beta
@RequiredArgsConstructor( access = AccessLevel.PACKAGE )
public abstract class AbstractODataParameters
{
    @Getter( AccessLevel.PACKAGE )
    @Nonnull
    private final Map<String, Expressions.OperandSingle> parameters = new LinkedHashMap<>();

    /**
     * The {@link ODataProtocol} these parameters should conform to.
     */
    @Getter
    @Nonnull
    private final ODataProtocol protocol;

    /**
     * Add a parameter.
     *
     * @param parameterName
     *            Name of the entity property or function parameters.
     * @param value
     *            Property value, assumed to be a primitive.
     * @param <PrimitiveT>
     *            Type of the primitive value.
     * @throws IllegalArgumentException
     *             When a parameter by that idenfitier already exists or primitive type is not supported.
     */
    <PrimitiveT> void addParameterInternal( @Nonnull final String parameterName, @Nullable final PrimitiveT value )
    {
        if( parameters.containsKey(parameterName) ) {
            throw new IllegalArgumentException(
                "Cannot add parameter \"" + parameterName + "\": A parameter by that name already exists.");
        }
        parameters.put(parameterName, Expressions.createOperand(value));
    }

    /**
     * Convenience method to add multpiple parameters at once.
     * 
     * @throws IllegalArgumentException
     *             When one of the values is null.
     * 
     * @see #addParameterInternal(String, Object)
     */
    void addParameterSetInternal( @Nonnull final Map<String, Object> properties )
    {
        properties.forEach(this::addParameterInternal);
    }

    /**
     * Serializes all parameters into an <strong>encoded</strong> URL path segment. The format is as follows:
     * <p>
     * <ul>
     * <li>An empty set of parameters will yield {@code ()}</li>
     * <li>A single parameter entry will yield {@code (value)}</li>
     * <li>Multiple of parameters will yield {@code (key1=val,key2=val)}</li>
     * </ul>
     * </p>
     * 
     * @return Encoded URL string representation of the parameters.
     */
    @Nonnull
    public String toEncodedString()
    {
        return toStringInternal(true, true);
    }

    /**
     * Serializes all parameters into an <strong>unencoded</strong> URL path segment. The format is as follows:
     * <p>
     * <ul>
     * <li>An empty set of parameters will yield {@code ()}</li>
     * <li>A single parameter entry will yield {@code (value)}</li>
     * <li>Multiple of parameters will yield {@code (key1=val,key2=val)}</li>
     * </ul>
     * </p>
     *
     * @return String representation of the parameters.
     */
    @Nonnull
    @Override
    public String toString()
    {
        return toStringInternal(false, true);
    }

    @Nonnull
    String toStringInternal( final boolean applyEncoding, final boolean applyShortFormat )
    {
        return toStringInternal(applyEncoding, applyShortFormat, false);
    }

    @Nonnull
    String toStringInternal( final boolean applyEncoding, final boolean applyShortFormat, final boolean queryFormat )
    {
        final Function<String, String> encoder;
        if( applyEncoding ) {
            encoder = queryFormat ? ODataUriFactory::encodeQuery : ODataUriFactory::encodePathSegment;
        } else {
            encoder = Function.identity();
        }
        if( applyShortFormat && !queryFormat && parameters.size() == 1 ) {
            final Expressions.OperandSingle singleValue = parameters.values().iterator().next();
            String parameterValue = singleValue.getExpression(protocol);
            parameterValue = encoder.apply(parameterValue);
            parameterValue = String.format("(%s)", parameterValue);

            return parameterValue;
        }
        final String keyDelimiter = queryFormat ? "&" : ",";
        String keys =
            parameters
                .entrySet()
                .stream()
                .map(param -> Tuple.of(param.getKey(), param.getValue()))
                .map(param -> param.map2(val -> val.getExpression(protocol)))
                .map(param -> param.map2(encoder))
                .map(param -> param.apply(( key, val ) -> String.format("%s=%s", key, val)))
                .collect(Collectors.joining(keyDelimiter));

        if( !queryFormat ) {
            keys = String.format("(%s)", keys);
        }
        return keys;
    }
}
