/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.quality;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.UUID;

import javax.annotation.Nonnull;

import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVPrinter;

import lombok.RequiredArgsConstructor;

/**
 * Writes a given set of line entries into a csv file.
 */
@RequiredArgsConstructor
class ServiceAuditReportWriter
{
    private static final String DEFAULT_REPORTS_DIRECTORY = "../s4hana_pipeline/reports/service_audits";

    @Nonnull
    private final Path reportPath;

    /**
     * Creates a new writer for a file based on the given report name.
     * <p>
     * The report file will be located under
     * <i>{@value DEFAULT_REPORTS_DIRECTORY}/&lt;reportName&gt;/&lt;someUUID&gt;.log</i>
     * 
     * @param reportName
     *            The name of the report, used to generate a unique file name.
     */
    ServiceAuditReportWriter( @Nonnull final String reportName )
    {
        reportPath = Paths.get(DEFAULT_REPORTS_DIRECTORY + "/" + reportName + "_" + UUID.randomUUID() + ".log");
    }

    void writeLine( final String lineIdentifier, final String... lineValues )
    {
        if( !Files.exists(reportPath) ) {
            try {
                Files.createDirectories(reportPath.getParent());
                Files.createFile(reportPath);
            }
            catch( final IOException e ) {
                throw new QualityListenerException("Error: Creating " + reportPath + " failed.", e);
            }
        }

        final StringBuilder builder = new StringBuilder();
        try( CSVPrinter csvPrinter = new CSVPrinter(builder, CSVFormat.DEFAULT) ) {
            final Collection<String> lineItems = new ArrayList<>(lineValues.length + 1);
            lineItems.add(lineIdentifier);
            lineItems.addAll(Arrays.asList(lineValues));
            csvPrinter.printRecord(lineItems);

            Files
                .write(
                    reportPath,
                    Collections.singletonList(builder),
                    StandardCharsets.UTF_8,
                    StandardOpenOption.CREATE,
                    StandardOpenOption.APPEND);
        }
        catch( final IOException e ) {
            throw new QualityListenerException(e);
        }
    }
}
