/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.attachment;

import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>GetAttachmentCount</b> OData function import.
 *
 */
public class GetAttachmentCountFluentHelper
    extends
    SingleValuedFluentHelperFunction<GetAttachmentCountFluentHelper, AttachmentsCount, AttachmentsCount>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>GetAttachmentCount</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param linkedSAPObjectKey
     *            LinkedSAPObjectKey
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 90
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>LinkedSAPObjectKey</b>
     *            </p>
     * @param businessObjectTypeName
     *            BusinessObjectTypeName
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BusinessObjectTypeName</b>
     *            </p>
     * @param semanticObject
     *            SemanticObject
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SemanticObject</b>
     *            </p>
     */
    public GetAttachmentCountFluentHelper(
        @Nonnull final String servicePath,
        @Nullable final String businessObjectTypeName,
        @Nonnull final String linkedSAPObjectKey,
        @Nullable final String semanticObject )
    {
        super(servicePath);
        if( businessObjectTypeName != null ) {
            values.put("BusinessObjectTypeName", businessObjectTypeName);
        }
        values.put("LinkedSAPObjectKey", linkedSAPObjectKey);
        if( semanticObject != null ) {
            values.put("SemanticObject", semanticObject);
        }
    }

    /**
     * Creates a fluent helper object that will execute the <b>GetAttachmentCount</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param linkedSAPObjectKey
     *            LinkedSAPObjectKey
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 90
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>LinkedSAPObjectKey</b>
     *            </p>
     * @param businessObjectTypeName
     *            BusinessObjectTypeName
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BusinessObjectTypeName</b>
     *            </p>
     * @param semanticObject
     *            SemanticObject
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SemanticObject</b>
     *            </p>
     * @param sAPObjectType
     *            SAPObjectType
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 30
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SAPObjectType</b>
     *            </p>
     * @param sAPObjectNodeType
     *            SAPObjectNodeType
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 30
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SAPObjectNodeType</b>
     *            </p>
     */
    public GetAttachmentCountFluentHelper(
        @Nonnull final String servicePath,
        @Nullable final String businessObjectTypeName,
        @Nonnull final String linkedSAPObjectKey,
        @Nullable final String semanticObject,
        @Nullable final String sAPObjectType,
        @Nullable final String sAPObjectNodeType )
    {
        super(servicePath);
        if( businessObjectTypeName != null ) {
            values.put("BusinessObjectTypeName", businessObjectTypeName);
        }
        values.put("LinkedSAPObjectKey", linkedSAPObjectKey);
        if( semanticObject != null ) {
            values.put("SemanticObject", semanticObject);
        }
        if( sAPObjectType != null ) {
            values.put("SAPObjectType", sAPObjectType);
        }
        if( sAPObjectNodeType != null ) {
            values.put("SAPObjectNodeType", sAPObjectNodeType);
        }
    }

    @Override
    @Nonnull
    protected Class<AttachmentsCount> getEntityClass()
    {
        return AttachmentsCount.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "GetAttachmentCount";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpGet(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public AttachmentsCount executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
