/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.attachment;

import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>LinkAttachmentToSAPObject</b> OData function import.
 *
 */
public class LinkAttachmentToSAPObjectFluentHelper
    extends
    SingleValuedFluentHelperFunction<LinkAttachmentToSAPObjectFluentHelper, AttachmentHarmonizedOperation, AttachmentHarmonizedOperation>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>LinkAttachmentToSAPObject</b> OData function import with
     * the provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param archiveDocumentID
     *            ArchiveDocumentID
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ArchiveDocumentID</b>
     *            </p>
     * @param linkedSAPObjectKey
     *            LinkedSAPObjectKey
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 90
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>LinkedSAPObjectKey</b>
     *            </p>
     * @param harmonizedDocumentType
     *            HarmonizedDocumentType
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>HarmonizedDocumentType</b>
     *            </p>
     * @param sAPObjectType
     *            SAPObjectType
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 30
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SAPObjectType</b>
     *            </p>
     * @param sAPObjectNodeType
     *            SAPObjectNodeType
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 30
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SAPObjectNodeType</b>
     *            </p>
     * @param logicalDocument
     *            LogicalDocument
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 32
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>LogicalDocument</b>
     *            </p>
     */
    public LinkAttachmentToSAPObjectFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String linkedSAPObjectKey,
        @Nonnull final String harmonizedDocumentType,
        @Nonnull final String archiveDocumentID,
        @Nonnull final String logicalDocument,
        @Nullable final String sAPObjectNodeType,
        @Nonnull final String sAPObjectType )
    {
        super(servicePath);
        values.put("LinkedSAPObjectKey", linkedSAPObjectKey);
        values.put("HarmonizedDocumentType", harmonizedDocumentType);
        values.put("ArchiveDocumentID", archiveDocumentID);
        values.put("LogicalDocument", logicalDocument);
        if( sAPObjectNodeType != null ) {
            values.put("SAPObjectNodeType", sAPObjectNodeType);
        }
        values.put("SAPObjectType", sAPObjectType);
    }

    @Override
    @Nonnull
    protected Class<AttachmentHarmonizedOperation> getEntityClass()
    {
        return AttachmentHarmonizedOperation.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "LinkAttachmentToSAPObject";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public AttachmentHarmonizedOperation executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
