/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.bankdetail;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.bankdetail.selectable.BankDetailSelectable;

/**
 * Fluent helper to fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.bankdetail.BankDetail
 * BankDetail} entity using key fields. This fluent helper allows methods which modify the underlying query to be called
 * before executing the query itself.
 *
 */
public class BankDetailByKeyFluentHelper
    extends
    FluentHelperByKey<BankDetailByKeyFluentHelper, BankDetail, BankDetailSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.bankdetail.BankDetail BankDetail} entity with the
     * provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code BankDetail}
     * @param servicePath
     *            Service path to be used to fetch a single {@code BankDetail}
     * @param bankInternalID
     *            Bank Keys
     *            <p>
     *            Constraints: Not nullable, Maximum length: 15
     *            </p>
     * @param bankCountry
     *            Country/Region Key of Bank
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     */
    public BankDetailByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String bankCountry,
        final String bankInternalID )
    {
        super(servicePath, entityCollection);
        this.key.put("BankCountry", bankCountry);
        this.key.put("BankInternalID", bankInternalID);
    }

    @Override
    @Nonnull
    protected Class<BankDetail> getEntityClass()
    {
        return BankDetail.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
