/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.bankreconciliationstatement;

import java.net.URI;
import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>StartReconcile</b> OData function import.
 *
 */
public class StartReconcileFluentHelper
    extends
    SingleValuedFluentHelperFunction<StartReconcileFluentHelper, BankReconciliationItem, BankReconciliationItem>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>StartReconcile</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param companyCode
     *            Constraints: Not nullable, Maximum length: 4
     *            <p>
     *            Original parameter name from the Odata EDM: <b>CompanyCode</b>
     *            </p>
     * @param houseBankAccount
     *            Constraints: Not nullable, Maximum length: 5
     *            <p>
     *            Original parameter name from the Odata EDM: <b>HouseBankAccount</b>
     *            </p>
     * @param bankReconciliationDate
     *            Time Stamp
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BankReconciliationDate</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param houseBank
     *            Constraints: Not nullable, Maximum length: 5
     *            <p>
     *            Original parameter name from the Odata EDM: <b>HouseBank</b>
     *            </p>
     */
    public StartReconcileFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String companyCode,
        @Nonnull final String houseBank,
        @Nonnull final String houseBankAccount,
        @Nonnull final LocalDateTime bankReconciliationDate )
    {
        super(servicePath);
        values.put("CompanyCode", companyCode);
        values.put("HouseBank", houseBank);
        values.put("HouseBankAccount", houseBankAccount);
        values.put("BankReconciliationDate", bankReconciliationDate);
    }

    @Override
    @Nonnull
    protected Class<BankReconciliationItem> getEntityClass()
    {
        return BankReconciliationItem.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "StartReconcile";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public BankReconciliationItem executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
