/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.basicproductavailabilityinformation;

import java.net.URI;
import java.time.ZonedDateTime;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>DetermineAvailabilityAt</b> OData function import.
 *
 */
public class DetermineAvailabilityAtFluentHelper
    extends
    SingleValuedFluentHelperFunction<DetermineAvailabilityAtFluentHelper, AvailabilityRecord, AvailabilityRecord>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>DetermineAvailabilityAt</b> OData function import with
     * the provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param material
     *            Material
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Material</b>
     *            </p>
     * @param aTPCheckingRule
     *            Checking rule
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ATPCheckingRule</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param requestedUTCDateTime
     *            Date
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Precision: 7
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>RequestedUTCDateTime</b>
     *            </p>
     * @param supplyingPlant
     *            Plant
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SupplyingPlant</b>
     *            </p>
     */
    public DetermineAvailabilityAtFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String supplyingPlant,
        @Nonnull final String material,
        @Nonnull final String aTPCheckingRule,
        @Nonnull final ZonedDateTime requestedUTCDateTime )
    {
        super(servicePath);
        values.put("SupplyingPlant", supplyingPlant);
        values.put("Material", material);
        values.put("ATPCheckingRule", aTPCheckingRule);
        values.put("RequestedUTCDateTime", requestedUTCDateTime);
    }

    @Override
    @Nonnull
    protected Class<AvailabilityRecord> getEntityClass()
    {
        return AvailabilityRecord.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "DetermineAvailabilityAt";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpGet(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public AvailabilityRecord executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
