/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.batch;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.batch.selectable.BatchPlantSelectable;

/**
 * Fluent helper to fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.batch.BatchPlant
 * BatchPlant} entity using key fields. This fluent helper allows methods which modify the underlying query to be called
 * before executing the query itself.
 *
 */
public class BatchPlantByKeyFluentHelper
    extends
    FluentHelperByKey<BatchPlantByKeyFluentHelper, BatchPlant, BatchPlantSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.batch.BatchPlant BatchPlant} entity with the provided
     * key field values. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code BatchPlant}
     * @param material
     *            Material Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 18
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code BatchPlant}
     * @param plant
     *            Plant
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param batch
     *            Batch Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     */
    public BatchPlantByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String material,
        final String batch,
        final String plant )
    {
        super(servicePath, entityCollection);
        this.key.put("Material", material);
        this.key.put("Batch", batch);
        this.key.put("Plant", plant);
    }

    @Override
    @Nonnull
    protected Class<BatchPlant> getEntityClass()
    {
        return BatchPlant.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
