/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.batch;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.batch.selectable.BatchTextSelectable;

/**
 * Fluent helper to fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.batch.BatchText BatchText}
 * entity using key fields. This fluent helper allows methods which modify the underlying query to be called before
 * executing the query itself.
 *
 */
public class BatchTextByKeyFluentHelper
    extends
    FluentHelperByKey<BatchTextByKeyFluentHelper, BatchText, BatchTextSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.batch.BatchText BatchText} entity with the provided
     * key field values. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code BatchText}
     * @param material
     *            Material Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 18
     *            </p>
     * @param batchIdentifyingPlant
     *            Plant
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code BatchText}
     * @param batch
     *            Batch Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param language
     *            Language Key
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     */
    public BatchTextByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String material,
        final String batchIdentifyingPlant,
        final String batch,
        final String language )
    {
        super(servicePath, entityCollection);
        this.key.put("Material", material);
        this.key.put("BatchIdentifyingPlant", batchIdentifyingPlant);
        this.key.put("Batch", batch);
        this.key.put("Language", language);
    }

    @Override
    @Nonnull
    protected Class<BatchText> getEntityClass()
    {
        return BatchText.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
