/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.billofmaterials;

import java.math.BigDecimal;
import java.net.URI;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.CollectionValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>ExplodeBOM</b> OData function import.
 *
 */
public class ExplodeBOMFluentHelper
    extends
    CollectionValuedFluentHelperFunction<ExplodeBOMFluentHelper, DBomheaderforexplosionOut, List<DBomheaderforexplosionOut>>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>ExplodeBOM</b> OData function import with the provided
     * parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent helper
     * object.
     *
     * @param bOMItmQtyIsScrapRelevant
     *            Exceptions
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 1
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BOMItmQtyIsScrapRelevant</b>
     *            </p>
     * @param sparePartFltrType
     *            Spare Part Indicator
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 1
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SparePartFltrType</b>
     *            </p>
     * @param bOMExplosionLevel
     *            Explosion level
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 3, Scale: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BOMExplosionLevel</b>
     *            </p>
     * @param bOMExplosionIsLimited
     *            Limited Explosion
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BOMExplosionIsLimited</b>
     *            </p>
     * @param bOMExplosionApplication
     *            BOM Application
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BOMExplosionApplication</b>
     *            </p>
     * @param requiredQuantity
     *            Base quantity
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 13, Scale: 3
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>RequiredQuantity</b>
     *            </p>
     * @param material
     *            Material
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Material</b>
     *            </p>
     * @param billOfMaterialHeaderUUID
     *            ID hdr chge status
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: none
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BillOfMaterialHeaderUUID</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param plant
     *            Plant
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Plant</b>
     *            </p>
     * @param engineeringChangeDocument
     *            Change Number
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 12
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>EngineeringChangeDocument</b>
     *            </p>
     * @param billOfMaterialItemCategory
     *            Item Category
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 1
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BillOfMaterialItemCategory</b>
     *            </p>
     * @param bOMExplosionIsMultilevel
     *            Multi-lev.
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BOMExplosionIsMultilevel</b>
     *            </p>
     * @param billOfMaterialVariant
     *            Alternative BOM
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 2
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BillOfMaterialVariant</b>
     *            </p>
     * @param materialProvisionFltrType
     *            Mat. Provision Ind.
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 1
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>MaterialProvisionFltrType</b>
     *            </p>
     * @param bOMExplosionAssembly
     *            Material
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BOMExplosionAssembly</b>
     *            </p>
     * @param bOMExplosionDate
     *            Valid From
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BOMExplosionDate</b>
     *            </p>
     */
    public ExplodeBOMFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final UUID billOfMaterialHeaderUUID,
        @Nullable final String material,
        @Nullable final String plant,
        @Nullable final String billOfMaterialVariant,
        @Nullable final String bOMExplosionApplication,
        @Nullable final BigDecimal requiredQuantity,
        @Nullable final String engineeringChangeDocument,
        @Nullable final Boolean bOMExplosionIsLimited,
        @Nullable final String bOMItmQtyIsScrapRelevant,
        @Nullable final String billOfMaterialItemCategory,
        @Nullable final String bOMExplosionAssembly,
        @Nullable final LocalDateTime bOMExplosionDate,
        @Nullable final BigDecimal bOMExplosionLevel,
        @Nullable final Boolean bOMExplosionIsMultilevel,
        @Nullable final String materialProvisionFltrType,
        @Nullable final String sparePartFltrType )
    {
        super(servicePath);
        values.put("BillOfMaterialHeaderUUID", billOfMaterialHeaderUUID);
        if( material != null ) {
            values.put("Material", material);
        }
        if( plant != null ) {
            values.put("Plant", plant);
        }
        if( billOfMaterialVariant != null ) {
            values.put("BillOfMaterialVariant", billOfMaterialVariant);
        }
        if( bOMExplosionApplication != null ) {
            values.put("BOMExplosionApplication", bOMExplosionApplication);
        }
        if( requiredQuantity != null ) {
            values.put("RequiredQuantity", requiredQuantity);
        }
        if( engineeringChangeDocument != null ) {
            values.put("EngineeringChangeDocument", engineeringChangeDocument);
        }
        if( bOMExplosionIsLimited != null ) {
            values.put("BOMExplosionIsLimited", bOMExplosionIsLimited);
        }
        if( bOMItmQtyIsScrapRelevant != null ) {
            values.put("BOMItmQtyIsScrapRelevant", bOMItmQtyIsScrapRelevant);
        }
        if( billOfMaterialItemCategory != null ) {
            values.put("BillOfMaterialItemCategory", billOfMaterialItemCategory);
        }
        if( bOMExplosionAssembly != null ) {
            values.put("BOMExplosionAssembly", bOMExplosionAssembly);
        }
        if( bOMExplosionDate != null ) {
            values.put("BOMExplosionDate", bOMExplosionDate);
        }
        if( bOMExplosionLevel != null ) {
            values.put("BOMExplosionLevel", bOMExplosionLevel);
        }
        if( bOMExplosionIsMultilevel != null ) {
            values.put("BOMExplosionIsMultilevel", bOMExplosionIsMultilevel);
        }
        if( materialProvisionFltrType != null ) {
            values.put("MaterialProvisionFltrType", materialProvisionFltrType);
        }
        if( sparePartFltrType != null ) {
            values.put("SparePartFltrType", sparePartFltrType);
        }
    }

    @Override
    @Nonnull
    protected Class<DBomheaderforexplosionOut> getEntityClass()
    {
        return DBomheaderforexplosionOut.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "ExplodeBOM";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nonnull
    public List<DBomheaderforexplosionOut> executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeMultiple(destination);
    }

}
