/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.billofmaterials;

import java.net.URI;
import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>ReleaseVersionBOM</b> OData function import.
 *
 */
public class ReleaseVersionBOMFluentHelper
    extends
    SingleValuedFluentHelperFunction<ReleaseVersionBOMFluentHelper, BillOfMaterial, BillOfMaterial>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>ReleaseVersionBOM</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param stlnr
     *            Bill of Material
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 8
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Stlnr</b>
     *            </p>
     * @param billOfMaterialHeaderUUID
     *            ID hdr chge status
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: none
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BillOfMaterialHeaderUUID</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param versnlastind
     *            Latest Rel Version
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Versnlastind</b>
     *            </p>
     * @param stlty
     *            BOM category
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 1
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Stlty</b>
     *            </p>
     * @param versnst
     *            BOM Version Status
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 2
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Versnst</b>
     *            </p>
     * @param stlal
     *            Alternative BOM
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 2
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Stlal</b>
     *            </p>
     * @param bomVersn
     *            BOM Version
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BomVersn</b>
     *            </p>
     */
    public ReleaseVersionBOMFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final UUID billOfMaterialHeaderUUID,
        @Nullable final String stlnr,
        @Nullable final String stlty,
        @Nullable final String stlal,
        @Nullable final String bomVersn,
        @Nullable final String versnst,
        @Nullable final Boolean versnlastind )
    {
        super(servicePath);
        values.put("BillOfMaterialHeaderUUID", billOfMaterialHeaderUUID);
        if( stlnr != null ) {
            values.put("Stlnr", stlnr);
        }
        if( stlty != null ) {
            values.put("Stlty", stlty);
        }
        if( stlal != null ) {
            values.put("Stlal", stlal);
        }
        if( bomVersn != null ) {
            values.put("BomVersn", bomVersn);
        }
        if( versnst != null ) {
            values.put("Versnst", versnst);
        }
        if( versnlastind != null ) {
            values.put("Versnlastind", versnlastind);
        }
    }

    @Override
    @Nonnull
    protected Class<BillOfMaterial> getEntityClass()
    {
        return BillOfMaterial.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "ReleaseVersionBOM";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public BillOfMaterial executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
