/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.billofmaterialsv2;

import java.math.BigDecimal;
import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>ConvertItem</b> OData function import.
 *
 */
public class ConvertItemFluentHelper
    extends
    SingleValuedFluentHelperFunction<ConvertItemFluentHelper, MaterialBOMItem, MaterialBOMItem>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>ConvertItem</b> OData function import with the provided
     * parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent helper
     * object.
     *
     * @param headerChangeDocument
     *            Change Number
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>HeaderChangeDocument</b>
     *            </p>
     * @param bOMItemDescription
     *            Item Text
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BOMItemDescription</b>
     *            </p>
     * @param billOfMaterialComponent
     *            Material
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BillOfMaterialComponent</b>
     *            </p>
     * @param billOfMaterialVersion
     *            BOM Version
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BillOfMaterialVersion</b>
     *            </p>
     * @param billOfMaterial
     *            Bill of Material
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 8
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BillOfMaterial</b>
     *            </p>
     * @param purchasingGroup
     *            Purchasing Group
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 3
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PurchasingGroup</b>
     *            </p>
     * @param billOfMaterialItemNodeNumber
     *            Item node
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 8
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BillOfMaterialItemNodeNumber</b>
     *            </p>
     * @param material
     *            Material
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Material</b>
     *            </p>
     * @param billOfMaterialCategory
     *            BOM category
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 1
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BillOfMaterialCategory</b>
     *            </p>
     * @param materialGroup
     *            Material Group
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 9
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>MaterialGroup</b>
     *            </p>
     * @param materialPriceUnitQty
     *            Price unit
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 5, Scale: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>MaterialPriceUnitQty</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param plant
     *            Plant
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Plant</b>
     *            </p>
     * @param materialComponentPrice
     *            Price
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 12, Scale: 3
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>MaterialComponentPrice</b>
     *            </p>
     * @param billOfMaterialItemCategory
     *            Item Category
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 1
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BillOfMaterialItemCategory</b>
     *            </p>
     * @param currency
     *            Currency
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 5
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Currency</b>
     *            </p>
     * @param billOfMaterialVariant
     *            Alternative BOM
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BillOfMaterialVariant</b>
     *            </p>
     */
    public ConvertItemFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String billOfMaterial,
        @Nonnull final String billOfMaterialCategory,
        @Nonnull final String billOfMaterialVariant,
        @Nonnull final String billOfMaterialVersion,
        @Nonnull final String billOfMaterialItemNodeNumber,
        @Nonnull final String headerChangeDocument,
        @Nonnull final String material,
        @Nonnull final String plant,
        @Nullable final String billOfMaterialComponent,
        @Nullable final String billOfMaterialItemCategory,
        @Nullable final String bOMItemDescription,
        @Nullable final String purchasingGroup,
        @Nullable final String materialGroup,
        @Nullable final BigDecimal materialPriceUnitQty,
        @Nullable final String currency,
        @Nullable final BigDecimal materialComponentPrice )
    {
        super(servicePath);
        values.put("BillOfMaterial", billOfMaterial);
        values.put("BillOfMaterialCategory", billOfMaterialCategory);
        values.put("BillOfMaterialVariant", billOfMaterialVariant);
        values.put("BillOfMaterialVersion", billOfMaterialVersion);
        values.put("BillOfMaterialItemNodeNumber", billOfMaterialItemNodeNumber);
        values.put("HeaderChangeDocument", headerChangeDocument);
        values.put("Material", material);
        values.put("Plant", plant);
        if( billOfMaterialComponent != null ) {
            values.put("BillOfMaterialComponent", billOfMaterialComponent);
        }
        if( billOfMaterialItemCategory != null ) {
            values.put("BillOfMaterialItemCategory", billOfMaterialItemCategory);
        }
        if( bOMItemDescription != null ) {
            values.put("BOMItemDescription", bOMItemDescription);
        }
        if( purchasingGroup != null ) {
            values.put("PurchasingGroup", purchasingGroup);
        }
        if( materialGroup != null ) {
            values.put("MaterialGroup", materialGroup);
        }
        if( materialPriceUnitQty != null ) {
            values.put("MaterialPriceUnitQty", materialPriceUnitQty);
        }
        if( currency != null ) {
            values.put("Currency", currency);
        }
        if( materialComponentPrice != null ) {
            values.put("MaterialComponentPrice", materialComponentPrice);
        }
    }

    @Override
    @Nonnull
    protected Class<MaterialBOMItem> getEntityClass()
    {
        return MaterialBOMItem.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "ConvertItem";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public MaterialBOMItem executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
