/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.billofmaterialsv2;

import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>ReleaseVersionBOM</b> OData function import.
 *
 */
public class ReleaseVersionBOMFluentHelper
    extends
    SingleValuedFluentHelperFunction<ReleaseVersionBOMFluentHelper, MaterialBOM, MaterialBOM>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>ReleaseVersionBOM</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param material
     *            Material
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Material</b>
     *            </p>
     * @param billOfMaterialCategory
     *            BOM category
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 1
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BillOfMaterialCategory</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param billOfMaterialVersion
     *            BOM Version
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BillOfMaterialVersion</b>
     *            </p>
     * @param engineeringChangeDocument
     *            Change Number
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>EngineeringChangeDocument</b>
     *            </p>
     * @param plant
     *            Plant
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Plant</b>
     *            </p>
     * @param billOfMaterial
     *            Bill of Material
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 8
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BillOfMaterial</b>
     *            </p>
     * @param billOfMaterialVariant
     *            Alternative BOM
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>BillOfMaterialVariant</b>
     *            </p>
     */
    public ReleaseVersionBOMFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String billOfMaterial,
        @Nonnull final String billOfMaterialCategory,
        @Nonnull final String billOfMaterialVariant,
        @Nonnull final String billOfMaterialVersion,
        @Nonnull final String engineeringChangeDocument,
        @Nonnull final String material,
        @Nonnull final String plant )
    {
        super(servicePath);
        values.put("BillOfMaterial", billOfMaterial);
        values.put("BillOfMaterialCategory", billOfMaterialCategory);
        values.put("BillOfMaterialVariant", billOfMaterialVariant);
        values.put("BillOfMaterialVersion", billOfMaterialVersion);
        values.put("EngineeringChangeDocument", engineeringChangeDocument);
        values.put("Material", material);
        values.put("Plant", plant);
    }

    @Override
    @Nonnull
    protected Class<MaterialBOM> getEntityClass()
    {
        return MaterialBOM.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "ReleaseVersionBOM";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public MaterialBOM executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
