/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.bufferprofile;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.bufferprofile.selectable.ProfileDetailsSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.bufferprofile.ProfileDetails ProfileDetails} entity using
 * key fields. This fluent helper allows methods which modify the underlying query to be called before executing the
 * query itself.
 *
 */
public class ProfileDetailsByKeyFluentHelper
    extends
    FluentHelperByKey<ProfileDetailsByKeyFluentHelper, ProfileDetails, ProfileDetailsSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.bufferprofile.ProfileDetails ProfileDetails} entity
     * with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method
     * on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code ProfileDetails}
     * @param isMinOrderQtyConsidered
     *            Is Minimum Order Quantity Considered
     *            <p>
     *            Constraints: Not nullable, Maximum length: 1
     *            </p>
     * @param variabilityCode
     *            Variability Co-efficient Indicator
     *            <p>
     *            Constraints: Not nullable, Maximum length: 1
     *            </p>
     * @param replenishmentLeadTimeCode
     *            Decoupled Lead Time Indicator
     *            <p>
     *            Constraints: Not nullable, Maximum length: 1
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code ProfileDetails}
     * @param plant
     *            Plant
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param procurementType
     *            Procurement Type
     *            <p>
     *            Constraints: Not nullable, Maximum length: 1
     *            </p>
     */
    public ProfileDetailsByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String plant,
        final String procurementType,
        final String variabilityCode,
        final String replenishmentLeadTimeCode,
        final String isMinOrderQtyConsidered )
    {
        super(servicePath, entityCollection);
        this.key.put("Plant", plant);
        this.key.put("ProcurementType", procurementType);
        this.key.put("VariabilityCode", variabilityCode);
        this.key.put("ReplenishmentLeadTimeCode", replenishmentLeadTimeCode);
        this.key.put("IsMinOrderQtyConsidered", isMinOrderQtyConsidered);
    }

    @Override
    @Nonnull
    protected Class<ProfileDetails> getEntityClass()
    {
        return ProfileDetails.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
