/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.buffersizing;

import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.buffersizing.selectable.ActualStockSelectable;

/**
 * Fluent helper to fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.buffersizing.ActualStock
 * ActualStock} entity using key fields. This fluent helper allows methods which modify the underlying query to be
 * called before executing the query itself.
 *
 */
public class ActualStockByKeyFluentHelper
    extends
    FluentHelperByKey<ActualStockByKeyFluentHelper, ActualStock, ActualStockSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.buffersizing.ActualStock ActualStock} entity with the
     * provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code ActualStock}
     * @param mRPArea
     *            MRP Area
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param material
     *            Material Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 18
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code ActualStock}
     * @param plant
     *            Plant
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param bufferLevelStartDate
     *            Valid-From Date
     *            <p>
     *            Constraints: Not nullable, Precision: 0
     *            </p>
     */
    public ActualStockByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String material,
        final String plant,
        final String mRPArea,
        final LocalDateTime bufferLevelStartDate )
    {
        super(servicePath, entityCollection);
        this.key.put("Material", material);
        this.key.put("Plant", plant);
        this.key.put("MRPArea", mRPArea);
        this.key.put("BufferLevelStartDate", bufferLevelStartDate);
    }

    @Override
    @Nonnull
    protected Class<ActualStock> getEntityClass()
    {
        return ActualStock.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
