/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.buffersizing;

import java.net.URI;
import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>Suspend</b> OData function import.
 *
 */
public class SuspendFluentHelper
    extends
    SingleValuedFluentHelperFunction<SuspendFluentHelper, ProposalHeader, ProposalHeader>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>Suspend</b> OData function import with the provided
     * parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent helper
     * object.
     *
     * @param proposalRunSuspendedToDate
     *            Prpsl Run Susp End
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ProposalRunSuspendedToDate</b>
     *            </p>
     * @param mRPArea
     *            MRP Area
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>MRPArea</b>
     *            </p>
     * @param material
     *            Material
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Material</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param plant
     *            Plant
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Plant</b>
     *            </p>
     * @param proposalVersion
     *            Proposal Version
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 8
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ProposalVersion</b>
     *            </p>
     * @param proposalRunSuspendedFromDate
     *            Prpsl Run Susp Start
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ProposalRunSuspendedFromDate</b>
     *            </p>
     * @param stockLevelProposalStatus
     *            Proposal Status
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 1
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>StockLevelProposalStatus</b>
     *            </p>
     * @param proposalStartDate
     *            Proposal Valid From
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ProposalStartDate</b>
     *            </p>
     */
    public SuspendFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String material,
        @Nonnull final String plant,
        @Nonnull final String mRPArea,
        @Nonnull final LocalDateTime proposalStartDate,
        @Nonnull final String proposalVersion,
        @Nullable final String stockLevelProposalStatus,
        @Nullable final LocalDateTime proposalRunSuspendedFromDate,
        @Nullable final LocalDateTime proposalRunSuspendedToDate )
    {
        super(servicePath);
        values.put("Material", material);
        values.put("Plant", plant);
        values.put("MRPArea", mRPArea);
        values.put("ProposalStartDate", proposalStartDate);
        values.put("ProposalVersion", proposalVersion);
        if( stockLevelProposalStatus != null ) {
            values.put("StockLevelProposalStatus", stockLevelProposalStatus);
        }
        if( proposalRunSuspendedFromDate != null ) {
            values.put("ProposalRunSuspendedFromDate", proposalRunSuspendedFromDate);
        }
        if( proposalRunSuspendedToDate != null ) {
            values.put("ProposalRunSuspendedToDate", proposalRunSuspendedToDate);
        }
    }

    @Override
    @Nonnull
    protected Class<ProposalHeader> getEntityClass()
    {
        return ProposalHeader.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "Suspend";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public ProposalHeader executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
