/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesspartner;

import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesspartner.selectable.BPRelationshipSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesspartner.BPRelationship BPRelationship} entity
 * using key fields. This fluent helper allows methods which modify the underlying query to be called before executing
 * the query itself.
 *
 */
public class BPRelationshipByKeyFluentHelper
    extends
    FluentHelperByKey<BPRelationshipByKeyFluentHelper, BPRelationship, BPRelationshipSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesspartner.BPRelationship BPRelationship} entity
     * with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method
     * on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code BPRelationship}
     * @param servicePath
     *            Service path to be used to fetch a single {@code BPRelationship}
     * @param businessPartner2
     *            Business Partner Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param businessPartner1
     *            Business Partner Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param validityEndDate
     *            Validity Date (Valid To)
     *            <p>
     *            Constraints: Not nullable, Precision: 0
     *            </p>
     * @param relationshipNumber
     *            BP Relationship Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     */
    public BPRelationshipByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String relationshipNumber,
        final String businessPartner1,
        final String businessPartner2,
        final LocalDateTime validityEndDate )
    {
        super(servicePath, entityCollection);
        this.key.put("RelationshipNumber", relationshipNumber);
        this.key.put("BusinessPartner1", businessPartner1);
        this.key.put("BusinessPartner2", businessPartner2);
        this.key.put("ValidityEndDate", validityEndDate);
    }

    @Override
    @Nonnull
    protected Class<BPRelationship> getEntityClass()
    {
        return BPRelationship.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
