/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesspartner;

import java.time.LocalDateTime;
import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesspartner.selectable.BusinessPartnerRatingSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesspartner.BusinessPartnerRating
 * BusinessPartnerRating} entity using key fields. This fluent helper allows methods which modify the underlying query
 * to be called before executing the query itself.
 *
 */
public class BusinessPartnerRatingByKeyFluentHelper
    extends
    FluentHelperByKey<BusinessPartnerRatingByKeyFluentHelper, BusinessPartnerRating, BusinessPartnerRatingSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesspartner.BusinessPartnerRating
     * BusinessPartnerRating} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param bPRatingValidityEndDate
     *            Valid-to Date for Rating
     *            <p>
     *            Constraints: Not nullable, Precision: 0
     *            </p>
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code BusinessPartnerRating}
     * @param businessPartner
     *            Business Partner Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code BusinessPartnerRating}
     * @param businessPartnerRatingProcedure
     *            Rating Procedure
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     */
    public BusinessPartnerRatingByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String businessPartner,
        final String businessPartnerRatingProcedure,
        final LocalDateTime bPRatingValidityEndDate )
    {
        super(servicePath, entityCollection);
        this.key.put("BusinessPartner", businessPartner);
        this.key.put("BusinessPartnerRatingProcedure", businessPartnerRatingProcedure);
        this.key.put("BPRatingValidityEndDate", bPRatingValidityEndDate);
    }

    @Override
    @Nonnull
    protected Class<BusinessPartnerRating> getEntityClass()
    {
        return BusinessPartnerRating.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
