/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesspartner;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesspartner.selectable.CustomerCompanySelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesspartner.CustomerCompany CustomerCompany} entity
 * using key fields. This fluent helper allows methods which modify the underlying query to be called before executing
 * the query itself.
 *
 */
public class CustomerCompanyByKeyFluentHelper
    extends
    FluentHelperByKey<CustomerCompanyByKeyFluentHelper, CustomerCompany, CustomerCompanySelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesspartner.CustomerCompany CustomerCompany}
     * entity with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest}
     * method on the fluent helper object.
     *
     * @param companyCode
     *            Company Code
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code CustomerCompany}
     * @param servicePath
     *            Service path to be used to fetch a single {@code CustomerCompany}
     * @param customer
     *            Customer Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     */
    public CustomerCompanyByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String customer,
        final String companyCode )
    {
        super(servicePath, entityCollection);
        this.key.put("Customer", customer);
        this.key.put("CompanyCode", companyCode);
    }

    @Override
    @Nonnull
    protected Class<CustomerCompany> getEntityClass()
    {
        return CustomerCompany.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
