/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesspartner;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesspartner.selectable.CustomerCompanyTextSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesspartner.CustomerCompanyText CustomerCompanyText}
 * entity using key fields. This fluent helper allows methods which modify the underlying query to be called before
 * executing the query itself.
 *
 */
public class CustomerCompanyTextByKeyFluentHelper
    extends
    FluentHelperByKey<CustomerCompanyTextByKeyFluentHelper, CustomerCompanyText, CustomerCompanyTextSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businesspartner.CustomerCompanyText
     * CustomerCompanyText} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param companyCode
     *            Company Code
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code CustomerCompanyText}
     * @param servicePath
     *            Service path to be used to fetch a single {@code CustomerCompanyText}
     * @param language
     *            Language key
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     * @param longTextID
     *            Text ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param customer
     *            Customer Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     */
    public CustomerCompanyTextByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String customer,
        final String companyCode,
        final String language,
        final String longTextID )
    {
        super(servicePath, entityCollection);
        this.key.put("Customer", customer);
        this.key.put("CompanyCode", companyCode);
        this.key.put("Language", language);
        this.key.put("LongTextID", longTextID);
    }

    @Override
    @Nonnull
    protected Class<CustomerCompanyText> getEntityClass()
    {
        return CustomerCompanyText.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
