/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businessplace;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businessplace.selectable.BusinessPlaceSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businessplace.BusinessPlace BusinessPlace} entity using
 * key fields. This fluent helper allows methods which modify the underlying query to be called before executing the
 * query itself.
 *
 */
public class BusinessPlaceByKeyFluentHelper
    extends
    FluentHelperByKey<BusinessPlaceByKeyFluentHelper, BusinessPlace, BusinessPlaceSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.businessplace.BusinessPlace BusinessPlace} entity with
     * the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method on
     * the fluent helper object.
     *
     * @param companyCode
     *            Company Code
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code BusinessPlace}
     * @param servicePath
     *            Service path to be used to fetch a single {@code BusinessPlace}
     * @param businessPlaceID
     *            Business Place
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     */
    public BusinessPlaceByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String companyCode,
        final String businessPlaceID )
    {
        super(servicePath, entityCollection);
        this.key.put("CompanyCode", companyCode);
        this.key.put("BusinessPlaceID", businessPlaceID);
    }

    @Override
    @Nonnull
    protected Class<BusinessPlace> getEntityClass()
    {
        return BusinessPlace.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
